/*
 * FootballPack.C
 *
 *   Calculate probabilities and statistics for football-type pools.
 *
 * Copyright (C) 2005 Bryan Clair
 *
 * This file is part of CLOP.
 *
 * CLOP is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * CLOP is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with CLOP; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *
 * Revisions:
 *   1/9/06 added quick_picks() to speed up searching for optimal.
 *   1/16/06 reversed game order in printOutcome(), added spaces btw games
 */

#include <iostream>
#include <iomanip>
#include <assert.h>
#include <pthread.h>
#include <gsl/gsl_math.h>
#include <gsl/gsl_diff.h>
#include "PoolMath.h"
#include "FootballPack.h"

//
//  Outcome routines
//
bool gameOutcome(Outcome x, int g) {
  return (x & (1 << g));
}

// return the score (# of equal picks) for a pair of outcomes
inline int score(Outcome x, Outcome y, int g) {
  Outcome v = x^y;

  int c; // count of bits in v
  // bit counting method due to Brian Kernighan
  for (c = 0; v; c++)
      v &= v - 1; // clear the least significant bit set

  return g-c;
}

// scoreOutcome returns the score (# of equal picks) for a pair of
//  outcomes.  Public version is not inlined.
int scoreOutcome(Outcome x, Outcome y, int g) {
  return score(x,y,g);
}

// printOutcome displays an outcome with one char per game, the two
//  possibilities are given by the optional c1, c2 arguments
void printOutcome(Outcome x, int games, char c1, char c2) {
  for (int i = 0; i<games; i++)
    std::cout << (gameOutcome(x,i) ? c1 : c2) << ' ';
}

//
// ProbVec class
//
//   Vector of probabilities 0 <= p[i] <= 1
//

//
//  ProbVec::ProbVec
//     Constructors
//
ProbVec::ProbVec(int games) {
  g = games;
}
ProbVec::ProbVec(int games,double p) {
  g = games;
  for (int i=0; i<g; i++) prob[i]=p;
}
ProbVec::ProbVec(int games,double p[]) {
  g = games;
  for (int i=0; i<g; i++) prob[i]=p[i];
}
ProbVec::ProbVec(const ProbVec &v) {
  g = v.g;
  for (int i=0; i<g; i++) prob[i]=v.prob[i];
}

//
// ProbVec::operator=
//    Assignment operator
//
ProbVec& ProbVec::operator=(const ProbVec& v) {
  if (this != &v) {
    g = v.g;
    for (int i=0; i<g; i++) prob[i]=v.prob[i];
  }
  return *this;
}
    
//
// ProbVec::display
//    Put to cout
//
void ProbVec::display() const {
  for (int i=g-1; i>=0; i--) {
    std::cout << std::setw(8) << prob[i];
    if (i > 0) std::cout << ", ";
  }
}

//
// ProbVec::match
//    Calculate the probability that a choice of games with this
//    ProbVec will match s or less than s games of the Outcome X.
//
void ProbVec::match
(const Outcome X, const int s, double& equal, double& less) const
{
  double S[g+1];
  double sym[g+1];
  int i,j,k;
  Outcome mask;
  double p,pj;
  bool sign, startsign;

  // Calculate S[j] = sum(prob[i]^j)
  //
  S[0] = 1;
  for (j=1; j<=g; j++) S[j] = 0;
  for (i=0, mask = 1; i < g; i++, mask <<= 1) {
    p = (X & mask) ? (prob[i]) : (1-prob[i]);
    pj = 1;
    for (j=1; j<=g; j++) {
      // Add p^j to S[j].
      pj *= p;
      S[j] += pj;
    }
  }

  // Calculate sym[j], the symmetric poly of degree j.
  sym[0] = 1;
  startsign = true;
  for (j = 1; j<=g; j++, startsign = !startsign) {
    sign = startsign;
    sym[j] = 0;
    for (i=0; i<j; i++, sign = !sign) {
      if (sign) sym[j] += sym[i]*S[j-i];
      else sym[j] -= sym[i]*S[j-i];
    }
    sym[j] = sym[j]/j;
  }

  /* // Debug output S and sym
   *   for (j=0;j<=g;j++) {
   *     cout << "S"<<j<<"="<<S[j];
   *     cout << "\tsym"<<j<<"="<<sym[j];
   *     cout << endl;
   *   }  
   */

  // Calculate L(X,s)
  if (s == 0) less = 0;
  else {
    less = 1;
    sign = false;  // first term is negative
    for (k = s; k <= g; k++, sign = !sign) {
      p = s * PM_binom[k][s] * sym[k] / k;
      if (sign) less += p;
      else less -= p;
    }
  }

  // Calculate E(X,s)
  equal = 0;
  sign = true; // first term positive
  for (k = s; k <= g; k++, sign = !sign) {
    p = PM_binom[k][s] * sym[k];
    if (sign) equal += p;
    else equal -= p;
  }
}

//
// Functions to create picks from ProbVec
//
//    pick randomly with h2h probabilities
Outcome ProbVec::random_picks() const {
  Outcome mask, X = 0;
  for (int i=0, mask = 1; i < g; i++, mask <<= 1)
    if (rand_prob() < prob[i]) X |= mask;
  return X;
}

//    pick the favorite in every game
Outcome ProbVec::favorites() const {
  Outcome mask, X = 0;
  for (int i=0, mask = 1; i < g; i++, mask <<= 1)
    if (prob[i] >= .5) X |= mask;
  return X;
}

//    pick the underdog in every game
Outcome ProbVec::underdogs() const {
  Outcome mask, X = 0;
  for (int i=0, mask = 1; i < g; i++, mask <<= 1)
    if (prob[i] < .5) X |= mask;
  return X;
}

// quick_picks
//    A quick approximation to optimal picks using a linear cutoff.
//    The cutoff line is the line through (.5,.5) with slope
//    1 - N^(-CUTOFF_LINE_ALPHA)
//
const double CUTOFF_LINE_ALPHA = 0.195;

Outcome quick_picks(const int N,
			   const ProbVec& actual, const ProbVec& perceived) {
  Outcome mask, X = 0;
  double slope = 1 - pow(N,-CUTOFF_LINE_ALPHA);
  for (int i=0, mask = 1; i < actual.g; i++, mask <<= 1)
    if (actual.prob[i] >= (perceived.prob[i] - .5)*slope) X |= mask;
  return X;
}

//    pick the edge.. favorite if a >= p, else underdog
Outcome edge_picks(const ProbVec& actual, const ProbVec& perceived) {
  Outcome mask, X = 0;
  for (int i=0, mask = 1; i < actual.g; i++, mask <<= 1)
    if (actual.prob[i] >= perceived.prob[i]) X |= mask;
  return X;
}


//
// probability
//    Calculate the probability of an Outcome X occuring for ProbVec p.
//
double probability(const ProbVec &p, const Outcome X) {
  double pX = 1;
  Outcome mask;
  for (int i=0, mask = 1; i < p.games(); i++, mask <<= 1)
      pX *= ((X & mask) ? p[i] : (1-p[i]));
  return pX;
}

//
// Calculate the expected return on a bet b, given actual and perceived
// probabilities and number of opponents.
//
double expected(int N, const ProbVec &a, const ProbVec &p, Outcome b) {
  double total =0;
  double tie, win, Xval;
  Outcome X;
  int g = a.games();

  for (X = 0; X < (1 << g); X++) {
    // Calculate the contribution of this outcome
    p.match(X,score(b,X,g),tie,win);
      
    // Apply pooltie and weight by actual probability
    Xval = probability(a,X) * pooltie(N,win,tie);
    
    total += Xval;
  }
  
  return total;
}


// struct ex_param
//    Parameter structure for gsl call to expected()
// ex_gsl
//    Wrapper for gsl library call to expected()
//
struct ex_param {
  int N;
  ProbVec &a;
  ProbVec &p;
  Outcome b;
  bool use_a; // true for a, false for p
  int i;
};
double ex_gsl(double xi, void *val)
{
  ex_param *v = (ex_param *)val;
  assert((xi >= 0) && (xi <= 1));
  if (v->use_a) v->a[v->i] = xi;
  else v->p[v->i] = xi;
  return expected(v->N,v->a,v->p,v->b);
}

// dexpected_dxi
//    Calculate the partial derivative of the expected function
//    with respect to a_i or p_i.
//    Note a and p are passed by value, because we will mess up the local copies.
//
double dexpected_dxi(int N, ProbVec a, ProbVec p, Outcome b, bool use_a, int i)
{
  ex_param v = {N,a,p,b,use_a,i};

  gsl_function F;
  double result, abserr;
  const double h0 = 1e-8;

  F.function = &ex_gsl;
  F.params = &v;

  ProbVec &x = (use_a?a:p);

  // Calculate the partial derivative.
  // Three cases because the value of x[i] is undefined off of [0,1]
  if (x[i] < h0)
    gsl_diff_forward(&F, x[i], &result, &abserr);
  else if (1 - x[i] < h0)
    gsl_diff_backward(&F, x[i], &result, &abserr);
  else
    gsl_diff_central(&F, x[i], &result, &abserr);

  return result;
}

/*********************************************************************/
/*   Functions to calculate statistics                               */
/*********************************************************************/

////////////////////////////
// mean
////////////////////////////

//    Mean score if you choose an Outcome using A and compare to X.
double mean(const ProbVec& A, Outcome X) {
  double mX = 0;
  Outcome mask;
  for (int i=0, mask = 1; i < A.games(); i++, mask <<= 1)
    mX += ((X & mask) ? A[i] : (1-A[i]));
  return mX;
}  
 
//    Mean score if you choose an Outcome using A, an Outcome using P and compare.
//    (symmetric in A and P):
double mean(const ProbVec& A, const ProbVec& P) {
  double mP = 0;
  for (int i=0; i < A.games(); i++)
    mP += A[i]*P[i] + (1-A[i])*(1-P[i]);

  return mP;
}

////////////////////////////
// variance
////////////////////////////

// variance(A,X)
//    of the score if you choose an Outcome using A and compare to X.
double variance(const ProbVec& A, Outcome X) {
  double vX = 0;
  for (int i=0; i < A.games(); i++)
    vX += A[i]*(1-A[i]);
  return vX;
}
  

// variance(A,P)
//    of the score if you choose an Outcome using A and P and compare.
//    (symmetric in A and P)
double variance(const ProbVec& A, const ProbVec& P) {
  double vP = 0;
  for (int i=0; i < A.games(); i++)
    vP += ((A[i] + P[i] - 2*A[i]*P[i]) *
	   (1 - A[i] - P[i] + 2*A[i]*P[i]));
  return vP;
}

////////////////////////////
// covariance
////////////////////////////

// covariance(A,X1,X2)
//    of the scores of two players, making given X1 and X2.
double covariance(const ProbVec& A, Outcome X1, Outcome X2) {
  double cXX = 0;
  Outcome mask;
  Outcome v = X1^X2; // 1 in games where X1, X2 disagree
  for (int i=0, mask = 1; i < A.games(); i++, mask <<= 1)
    if (v & mask)
      cXX -= A[i]*(1-A[i]);
    else
      cXX += A[i]*(1-A[i]);
  return cXX;
}

// covariance(A,P,X)
//    of the scores of two players, one using randomized strategy P,
//    and one choosing Outcome X.
double covariance(const ProbVec& A, const ProbVec& P, Outcome X) {
  double cPX = 0;
  Outcome mask;
  for (int i=0, mask = 1; i < A.games(); i++, mask <<= 1)
    if (X & mask)
      cPX += A[i]*(1-A[i])*(2*P[i]-1);
    else
      cPX -= A[i]*(1-A[i])*(2*P[i]-1);
  return cPX;
}

// covariance(A,P1,P2)
//    of the scores of two players using randomized strategy P1 and P2
double covariance(const ProbVec& A, const ProbVec& P1, const ProbVec& P2) {
  double cPP = 0;
  for (int i=0; i < A.games(); i++)
    cPP += A[i]*(1-A[i])*(2*P1[i]-1)*(2*P2[i]-1);
  return cPP;
}

