/*
 * FootballPack.h
 *
 *   Calculate exact probabilities and expectations for football-type pools.
 *
 * Copyright (C) 2005 Bryan Clair
 *
 * This file is part of CLOP.
 *
 * CLOP is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * CLOP is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with CLOP; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *
 * Revisions:
 *   1/9/06 added quick_picks() to speed up searching for optimal.
 */

#ifndef FOOTBALLPACK_H
#define FOOTBALLPACK_H

/*********************************************************************/
/*   Outcome                                                         */
/*********************************************************************/
/*
 *    One possible result of a pool.
 *    Currently, an outcome is stored as the low g bits of an unsigned int.
 *    Because of this, the number of games needs to be explicitly specified,
 *    and is capped at FP_MAX_GAMES.
 *    In addition, routines assume that all unused bits are zero.
 *
 */
typedef unsigned int Outcome;

#define FP_MAX_GAMES (8*sizeof(Outcome))

// gameOutcome returns true if the "favorite" wins in this outcome
bool gameOutcome(Outcome x, int g);

// printOutcome displays an outcome with one char per game, the two
//  possibilities are given by the optional c1, c2 arguments
void printOutcome(Outcome x, int games, char c1='1', char c2='2');

// scoreOutcome returns the score (# of equal picks) for a pair of
//  outcomes
int scoreOutcome(Outcome x, Outcome y, int g);

/*********************************************************************/
/*   ProbVec                                                         */
/*********************************************************************/
/*
 * Vector of probabilities 0 <= p[i] <= 1
 *
 */
class ProbVec {
public:
  ProbVec(int games = 0);
  ProbVec(int games, double p);
  ProbVec(int games, double p[]);
  ProbVec(const ProbVec&);
  ProbVec& operator=(const ProbVec&);

  inline int games(void) const { return g; }
  inline double& operator[](int i) { return prob[i]; }
  inline double operator[](int i) const { return prob[i]; }

  // Calculate the probability that a choice of games with this probability
  // will match s or less than s games of the Outcome X.
  void match(const Outcome X, const int s, double& equal, double& less) const;

  // Output for debugging
  void display() const;

  // Functions to create picks
  Outcome random_picks() const;
  //    pick randomly with h2h probabilities
  Outcome favorites() const;
  //    pick the favorite in every game
  Outcome underdogs() const;
  //    pick the underdog in every game
  friend Outcome edge_picks(const ProbVec& actual, const ProbVec& perceived);
  //    pick the edge.. favorite if a >= p, else underdog
  friend Outcome quick_picks(const int N,
			     const ProbVec& actual, const ProbVec& perceived);
  //    A quick approximation to optimal picks using a linear cutoff.

private:
  int g;
  double prob[FP_MAX_GAMES];
};

/*********************************************************************/
/*   Discrete (i.e. precise) calculations                            */
/*********************************************************************/

// expected
//    Calculate the exact expected return on a bet b, given actual
//    and perceived probabilities and number of opponents.
double expected(int N, const ProbVec &a, const ProbVec &p, Outcome b);

// dexpected_dxi
//    Calculate the partial derivative of the expected function
//    with respect to a_i or p_i (depending on use_a=true or false).
//    Note a and p are passed by value, because we will mess up the local copies.
//
double dexpected_dxi(int N, ProbVec a, ProbVec p, Outcome b, bool use_a, int i);

// probability
//    Calculate the probability of an Outcome X occuring for ProbVec p.
//    In the past, this was written p^X
double probability(const ProbVec& p, Outcome X);

/*********************************************************************/
/*   Functions to calculate statistics                               */
/*********************************************************************/

////////////////////////////
// mean
////////////////////////////

double mean(const ProbVec& A, Outcome X);
//    Mean score if you choose an Outcome using A and compare to X.

double mean(const ProbVec& A, const ProbVec& P);
//    Mean score if you choose an Outcome using A, an Outcome using P and compare.
//    (symmetric in A and P):

////////////////////////////
// variance
////////////////////////////

double variance(const ProbVec& A, Outcome X);
//    of the score if you choose an Outcome using A and compare to X.
//    (does not actually depend on X)

double variance(const ProbVec& A, const ProbVec& P);
//    of the score if you choose an Outcome using A and P and compare.
//    (symmetric in A and P)

////////////////////////////
// covariance
////////////////////////////

double covariance(const ProbVec& A, Outcome X1, Outcome X2);
//    of the scores of two players, making given X1 and X2.

double covariance(const ProbVec& A, const ProbVec& P, Outcome X);
//    of the scores of two players, one using randomized strategy P,
//    and one choosing Outcome X.

double covariance(const ProbVec& A, const ProbVec& P1, const ProbVec& P2);
//    of the scores of two players using randomized strategy P1 and P2

#endif
