/*
 * FootballPool.C
 *
 *   Manage data in a football pool.
 *
 * Copyright (C) 2005 Bryan Clair
 *
 * This file is part of CLOP.
 *
 * CLOP is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * CLOP is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with CLOP; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *
 * Revisions:
 *   9/16/06 - Added is_whitespace_string() to better handle blank lines
 */

#include <istream>
#include <ostream>
#include <string>
#include <sstream>
#include <ctype.h>

#include "FootballPack.h"
#include "FootballPool.h"
#include "Normal.h"

using namespace std;

//
// Constructors
//
FootballPool::FootballPool() {
  g = 0;
}

FootballPool::FootballPool(const FootballPool& pool) {
  g = pool.g;
  team1title = pool.team1title;
  team2title = pool.team2title;
  Atitle = pool.Atitle;
  Ptitle = pool.Ptitle;
  A = pool.A;
  P = pool.P;
  for (int i=0;  i<g; i++) {
    team1names[i] = pool.team1names[i];
    team2names[i] = pool.team2names[i];
  }
}

//
// Destructor
//
FootballPool::~FootballPool() {
}

//
// is_whitespace_string
//    return true if string is entirely whitespace
//
bool is_whitespace_string(string s) {
    for (int i=0; i<s.length(); i++) {
      if (!isspace(s[i])) return false;
    }
  return true;
}

//
// read_one_line
//    read one line of a pool file, skipping comment lines
//    and blank lines
//    Reads from file f and sets linestream to the line read.
//    Returns false for end-of-file, throws an exception for
//    read errors.
//
bool read_one_line(istream &f, istringstream &linestream) {
  string line;
  do {
    if (!getline(f,line)) {
      if (f.eof()) return false;
      else throw FP_read_error("read error");
    }
  } while (line[0] == '#' || is_whitespace_string(line));
  linestream.clear();
  linestream.str(line);
  return true;
}

// read
//    read pool data in ASCII format from a stream.
//    Uses columns with names actual_type and perceived_type
//    as actual and perceived probabilities, or if the
//    type names are NULL, uses column 3 for actual and
//    column 4 for perceived.
//    may throw an exception of type FP_read_error.
//
void FootballPool::read(istream& f,
			const string actual_type,
			const string perceived_type)
{
  istringstream line("");

  const int max_cols=16;
  int cols=0, acol = 2, pcol = 3, col;
  string col_title[max_cols+1];

  // Read header row, parse columns

  if (!read_one_line(f,line))
    throw FP_read_error("failed to read header line");

  while (line >> col_title[cols]) {
    cols++;
    if (cols > max_cols)
      throw FP_read_error("too many columns");
  }
  if (cols < 3) throw FP_read_error("too few columns");

  team1title = col_title[0];
  team2title = col_title[1];

  if (actual_type != "") {
    // search for it
    acol = 0;
    while (actual_type != col_title[acol]) {
      acol++;
      if (acol == cols)
	throw FP_read_error("specified header not found");
    }
  }
  Atitle = col_title[acol];

  if (perceived_type != "") {
    // search for it
    pcol = 0;
    while (perceived_type != col_title[pcol]) {
      pcol++;
      if (pcol == cols)
	throw FP_read_error("specified header not found");
    }
  }
  Ptitle = col_title[pcol];
  
  if (pcol < 2 || acol < 2)
    throw FP_read_error("header not for numeric column");


  // Read team & probability data
  double a[FP_MAX_GAMES], p[FP_MAX_GAMES], vals[max_cols];

  g = 0;
  while (read_one_line(f,line)) {
    if (g == FP_MAX_GAMES)
      throw FP_read_error("# of games exceeds maximum");
    line >> team1names[g] >> team2names[g];
    for (int c=2; c < cols; c++) {
      line >> vals[c];
      if (!line)
	throw FP_read_error("missing probability value");
      if (vals[c] < 0 || vals[c] > 1)
	throw FP_read_error("probability out of range");
    }
    a[g] = vals[acol];
    p[g] = vals[pcol];
    g++;
  }

  if (g == 0)
    throw FP_read_error("no pool data");

  // Create pool data structures
  A = ProbVec(g,a);
  P = ProbVec(g,p);
}

// write
//    write pool data in ASCII format to a stream.
//
void FootballPool::write(ostream& f) {
  f << team1title << '\t' << team2title << '\t'
    << Atitle << '\t' << Ptitle << endl;
  for (int i=0; i<g; i++) {
    f << team1names[i] << '\t' << team2names[i] << '\t'
      << A[i] << '\t' << P[i] << endl;
  }
}

// readOutcome
//    read an Outcome from a stream
//    may throw an exception of type FP_read_error.
Outcome FootballPool::readOutcome(istream& f) {
  Outcome X = 0;
  string t;
  for (int i=0; i<g; i++) {
    if (!(f >> t)) throw FP_read_error();
    if (t == team1names[i])
      X |= (1 << i);
    else if (t != team2names[i])
      throw FP_read_error(t + " failed to match a team");
  }
  return X;
}

// displayOutcome
//    print an Outcome to cout using teamname data
void FootballPool::displayOutcome(Outcome X) const {
  for (int i = 0; i<g; i++)
    cout << ((gameOutcome(X,i)) ? team1names[i] : team2names[i]) << " ";
}

// expected
//    Calculate the expected return on a bet b in this pool, given
//    number of opponents.
double FootballPool::expected(int N, Outcome b) {
  return ::expected(N, A, P, b);
}

// expected
//    Calculate the expected return on a bet b in this pool, given
//    number of opponents.
double FootballPool::smooth_expected(int N, Outcome b, Normal_integrator *nit) {
  if (nit == NULL) {
    Normal_integrator mynit;
    return mynit.expect_integrate(N,mean(A,P),variance(A,P),
				  mean(A,b), variance(A,b),
				  covariance(A,P,P),
				  covariance(A,P,b));
  } else {
    return nit->expect_integrate(N,mean(A,P),variance(A,P),
				 mean(A,b), variance(A,b),
				 covariance(A,P,P),
				 covariance(A,P,b));
  }
}

////////////////////
// FP_read_error
// exception class
////////////////////
FP_read_error::FP_read_error(string s) {
  err = "Pool read error";
  if (s != "") err = err + ": " + s;
}

void FP_read_error::display() const {
  cerr << err << endl;
}

