/*
 * FootballPool.h
 *
 *   Manage data in a football pool.
 *
 * Copyright (C) 2005 Bryan Clair
 *
 * This file is part of CLOP.
 *
 * CLOP is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * CLOP is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with CLOP; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *
 */

#ifndef FOOTBALLPOOL_H
#define FOOTBALLPOOL_H

#include <istream>
#include <ostream>
#include <iostream>
#include <string>

#include "FootballPack.h"
#include "Normal.h"

/*********************************************************************/
/*  Football Pool I/O Format                                         */
/*********************************************************************/
/*
 * + The first entry in the file is the number of games in the pool.
 * + This is followed by a header record, the specified number of game records,
 *   and the single character '.'
 * + Each record has from 3 to 16 whitespace separated fields
 *     Team 1 Name
 *     Team 2 Name
 *     One or more of : Team 1 probability of winning.
 * + The first record is the header record, and gives text names for the
 *   fields (for example "HOME" or "FAVORITE" or "MASSEY").
 *
 * By default, the 3rd and 4th field are used as the actual and perceived
 * probability of Team 1 winning.
 *
 *********************************************************************/

/*********************************************************************/
/*   FootballPool                                                    */
/*********************************************************************/
/*
 */
class FootballPool {
 public:
  FootballPool();
  FootballPool(const FootballPool&);
  ~FootballPool();

  int games() const { return g; }

  const ProbVec& actual() const { return A; }
  const ProbVec& perceived() const { return P; }
  const std::string& actual_title() const { return Atitle; }
  const std::string& perceived_title() const { return Ptitle; }

  // read and display pool
  //    routines read and write pool data in ASCII format to a stream.
  //    read may throw an exception of type FP_read_error.
  //
  void read(std::istream& f, const std::string actual_type = "",
	    const std::string perceived_type = "");
  void write(std::ostream& f);

  // displayOutcome
  //    print an Outcome to cout using teamname data
  void displayOutcome(Outcome) const;

  // readOutcome
  //    read an Outcome from a stream
  //    may throw an exception of type FP_read_error.
  Outcome readOutcome(std::istream& f);

  // expected
  //    Calculate the expected return on a bet b in this pool, given
  //    number of opponents.
  double expected(int N, Outcome b);

  // smooth_expected
  //    Calculate the approximate expected return on a bet b in this pool,
  //    given number of opponents.
  double smooth_expected(int N, Outcome b, Normal_integrator *nit = NULL);
  
 private:
  int g;
  std::string team1title, team2title, Atitle, Ptitle;
  ProbVec A, P;
  std::string team1names[FP_MAX_GAMES], team2names[FP_MAX_GAMES];
};

//
// FP_read_error
//    class for format errors when reading pool datafiles and Outcomes
//
class FP_read_error {
 public:
  FP_read_error(std::string s="");
  void display() const;
 private:
  std::string err;
};

#endif
