/*
 * Normal.C
 *
 *    Use a normal approximation to calculate expected values of bets.
 *    Needs:
 *      # of opponents
 *      mean and variance of one opponent
 *      covariance between two opponents
 *      your bet's mean and variance
 *      covariance of your bet and one opponent
 *
 * Copyright (C) 2005 Bryan Clair
 *
 * This file is part of CLOP.
 *
 * CLOP is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * CLOP is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with CLOP; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *
 */

#include <iostream>
#include <iomanip>
#include <limits>
#include <cassert>
#include <gsl/gsl_math.h>
#include <gsl/gsl_sf.h>
#include <gsl/gsl_integration.h>
using namespace std;

#include "Normal.h"

//
// double log_CDF(double x)
//   Computes log(CDF(x)) with accuracy for all values of x
//   CDF is the Cumulative Distribution Function for a normalized random variable
//   CDF(x) = \Phi(x) = \int_{-infty}^{x} Z(x) = .5*(1 + erf(x/sqrt(2))
//          = 1 - Q(x)
//   0 < CDF(x) < 1, so -infty < log(CDF(x)) < 0
//
double log_CDF(double x) {
  if (x > 0)
    return gsl_sf_log_1plusx(-erfc(x/M_SQRT2)/2);
  else // (x <= 0)
    return gsl_sf_log_erfc(-x/M_SQRT2) - M_LN2;
}

//
// double log_exp_x_m1(double x)
//   Computes log(exp(x)-1) for x > 0 in a way that is stable for all x.
//
double log_exp_x_m1(double x) {
  static const double big = -log(numeric_limits<double>::epsilon());
  if (x > big)
    return x;
  else
    return log(gsl_sf_expm1(x));
  // which is log(x) for x < epsilon.
}

//
// double win_distribution(double N, double x, double y)
//    For x > y, computes 
//                      N+1          N+1
//                CDF(x)    -  CDF(y) 
//                ----------------------
//                   CDF(x) - CDF(y)
//
double win_distribution(double N, double x, double y) {
  assert(x > y);

  double lCDFx = log_CDF(x), lCDFy = log_CDF(y);

  if (lCDFx - lCDFy < numeric_limits<double>::epsilon()/N)
    return (N+1)*exp(N*lCDFy);
  else
    return exp(N*lCDFy
	       + log_exp_x_m1((N+1)*(lCDFx - lCDFy))
	       - log_exp_x_m1(lCDFx - lCDFy) );
}


//
// Smooth expectations distribution function
//
double sed_f(double y, void *pv)
{
  Normal_integrator::int_params *p = (Normal_integrator::int_params *) pv;

  // gsl_sf_erf_Q computes the upper tail of the Gaussian probability function:
  //    Q(x) = (1/\sqrt{2\pi}) \int_x^\infty dt \exp(-t^2/2)
  // gsl_sf_erf_Z computes the Gaussian probability density function
  //    Z(x) = (1/\sqrt{2\pi}) \exp(-x^2/2).  

  // Old way, using pooltie():
  // double phiplus = 1-gsl_sf_erf_Q(p->m*y + p->bplus);
  // double phimins = 1-gsl_sf_erf_Q(p->m*y + p->bmins);
  // return pooltie_alt(p->N,phiplus,phimins)*gsl_sf_erf_Z(y);

  // New way, using win_distribution():
  //    The new way appears to make no difference.  We should test the two ways
  //    for speed, too.
  return win_distribution(p->N, p->m*y + p->bplus, p->m*y + p->bmins)*gsl_sf_erf_Z(y);
}

//
// Normal_integrator
//
Normal_integrator::Normal_integrator()
{
  // gsl library functions are advertised as threadsafe
  workspace = gsl_integration_workspace_alloc(BISECTION_LIMIT);

  params = new int_params;
  sed_F.function = &sed_f;
  sed_F.params = params;
}

//
// ~Normal_integrator
//
Normal_integrator::~Normal_integrator()
{
  gsl_integration_workspace_free(workspace);
  delete params;
}

//
// double Normal_integrator::expect_integrate
//
//    Opponents scores are random variables X1....XN
//    Our score is random variable Y.
//    Inputs are statistics:
//        muX = mean(Xi)
//        sig2X = variance(Xi)
//        muY = mean(Y)
//        sig2Y = variance(Y)
//        vXX = covariance(Xi,Xj)
//        vXY = covariance(Xi,Y)
//
//    Requires (sig2X > vXX) && (v(Xi-Y,Xj-Y) = vXX + sig2Y - 2*vXY >= 0)
//
//    Returns the smoothly approximated expected value for our bet
//    in a tournament with N opponents, with a payoff of (N+1)/k
//    for a k-way tie for first place.
//
double Normal_integrator::expect_integrate
(int N,
 double muX, double sig2X,
 double muY, double sig2Y,
 double vXX, double vXY)
{
  // Convert to statistics for random variables Di = Xi - Y
  assert(sig2X > vXX);

  double mu = muX - muY;
  double sig2 = sig2X + sig2Y - 2*vXY;
  double c = vXX + sig2Y - 2*vXY;

  assert(c >= 0);

  // Set up parameters for integration
  params->N = N;

  // Precompute what we can: slope and intercept of the two linear
  // transformations of y
  params->m = -sqrt(c/(sig2 - c));
  params->bplus = (.5-mu)/sqrt(sig2 - c);
  params->bmins = (-.5-mu)/sqrt(sig2 - c);

  // Since f is bounded by N+1, integrating against 
  //   the Gaussian is pointless outside [-cutoff,cutoff]
  double cutoff = sqrt(2*(log((double)N+1)-log(ACCURACY_GOAL)));

  /*
  // Debugging help
  double y;
  for (y = -cutoff; y <= cutoff; y=y+1) {
    cout << "(" << y << "," << sed_F.function(y,sed_F.params) << ")\n";
  }
  */

  double result, abserr;

  /*
  // Non-adaptive integral.  Should be faster, but has failed to converge with
  // reasonable dta.
  size_t evals;
  gsl_integration_qng(&sed_F,-cutoff,cutoff,EPSILON,ACCURACY_GOAL,&result,&abserr,&evals);
  cout << "function evaluations: " << evals << endl;
  */

  // Adaptive integral.  GSL_INTEG_GAUSS31 could also be ..15, 21, 31, 41, 51, 61 with
  // high rules giving better accuracy for smooth functions, low rules working better
  // for functions with local difficulties.
  gsl_integration_qag(&sed_F,-cutoff,cutoff,0,ACCURACY_GOAL,BISECTION_LIMIT,
  		      GSL_INTEG_GAUSS31,workspace,&result,&abserr);

  return result;
}

