/*
 * Normal.h
 *
 *    Use a normal approximation to calculate expected values of bets.
 *    Needs:
 *      # of opponents
 *      mean and variance of one opponent
 *      covariance between two opponents
 *      your bet's mean and variance
 *      covariance of your bet and one opponent
 *
 *  Programs compiled with Normal.o need to be linked with
 *      -lgsl -lgslcblas -lm
 *
 * Copyright (C) 2005 Bryan Clair
 *
 * This file is part of CLOP.
 *
 * CLOP is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * CLOP is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with CLOP; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *
 */

#ifndef _NORMAL_
#define _NORMAL_

#include <gsl/gsl_integration.h>

//
// Try to get this much accuracy (but no promises)
//
#define ACCURACY_GOAL 1e-8

//
// Number of bisections for adaptive integration
//
#define BISECTION_LIMIT 1000

//
// class Normal_integrator
//
//   Performs the numerical integration and calculations
//   needed for finding approximate expected values of bets.
//
//   Constructor allocates & initializes space used by GSL
//   for numeric integration.
//
//   double expect_integrate
//    Returns the smoothly approximated expected value for a bet
//    in a pool with the standard payoff scheme (N+1 pot, ties split)
//    Inputs:
//        N = number of opponents
//        muX = mean(Xi)
//        sig2X = variance(Xi)
//        muY = mean(Y)
//        sig2Y = variance(Y)
//        vXX = covariance(Xi,Xj)
//        vXY = covariance(Xi,Y)
//    where opponent scores are random variables X1....XN and
//    our score is random variable Y.
//    Requires
//       (sig2X > vXX) && (v(Xi-Y,Xj-Y) = vXX + sig2Y - 2*vXY >= 0)
//
class Normal_integrator {
 public:
  Normal_integrator();
  ~Normal_integrator();
  double expect_integrate(int N,
			  double muX, double sig2X,
			  double muY, double sig2Y,
			  double vXX, double vXY);
 private:
  //
  // Parameter structure for integration
  //
  struct int_params {
    int N;
    double m;
    double bplus;
    double bmins;
  };
  friend double sed_f(double y, void *pv);
  gsl_integration_workspace *workspace;
  int_params *params;
  gsl_function sed_F;
};

#endif
