/*
 * PoolMath.C
 *
 *    Utility mathematical functions.
 *
 * Copyright (C) 2005 Bryan Clair
 *
 * This file is part of CLOP.
 *
 * CLOP is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * CLOP is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with CLOP; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *
 */
#include <limits>
#include <stdlib.h>
#include <math.h>
#include <assert.h>
#include "PoolMath.h"
using namespace std;

#include "PoolMathBinom.C"

#define min(x,y) ((x) < (y) ? (x) : (y))
#define max(x,y) ((x) > (y) ? (x) : (y))

//
// double rand_prob()
//   Generate a random number uniformly distributed in [0,1].
// 
double rand_prob(void) {
  return random() / (double) RAND_MAX;
}

//
// bump_rand
//   Give a random double in the range [a,b] with a bump distribution
//   (more likely to be in the middle of the range)
//
double bump_rand(double a, double b) {
  return (a+b)/2 + ((b-a)/2)*tan(M_PI_4 * (2*(rand_prob()) - 1));
}

//
// distort_prob_bump
//   Randomly distort a probability.
//   Does a bump distribution centered on the current value
//   of p and with maximum possible width.
//
double distort_prob_bump (const double p) {
  double d;
  d = min(p,1-p);
  return bump_rand(p-d, p+d);
}


//
// distort_prob_uniform
//   Randomly distort a probability.
//   Uniform on the interval [p - epsilon, p + epsilon] /\ [0,1]
//
double distort_prob_uniform(const double p, const double epsilon) {
  double a = max(0,p-epsilon);
  double b = min(1,p+epsilon);
  return rand_prob()*(b-a) + a;
}

//
// pooltie
//
//   Computes        N+1   N+1
//              (x+e)  -  x           N      k    N-k
//    f(x,y) =  --------------  = sum   (x+e)  * x
//                    e             k=0
//
double pooltie(int N, double x, double e)
{
  //
  // If e is small, we replace f with it's e->0 limit.
  //
  if (e < numeric_limits<double>::epsilon())
    return (N+1)*pow(x+e,N);
  else
    return (pow(x+e,N+1) - pow(x,N+1))/e;
}

// pooltie_alt
//   Computes    N+1   N+1
//              x  -  y        N   k    N-k
//    f(x,y) =  -------  = sum    x  * y
//               x - y       k=0
//
double pooltie_alt(int N, double x, double y) {
  //
  // If x and y are close together, we replace f with its (x-y)->0 limit
  //
  if ((x-y) < numeric_limits<double>::epsilon())
    return (N+1)*pow(x,N);
  else
    return (pow(x,N+1) - pow(y,N+1))/(x - y);
}

