/*
 * TourneyBasics.C
 *
 *   Fundamental definitions for working with tournaments.
 *   Includes basic types and worth functions.
 *
 * Copyright (C) 2005 Bryan Clair
 *
 * This file is part of CLOP.
 *
 * CLOP is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * CLOP is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with CLOP; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *
 */

#include "TourneyBasics.h"

/*********************************************************************/
/* Team and Round manipulations
/*********************************************************************/

// rounds_from_teams
//       Calculate # of rounds needed for given # of teams
round_t rounds_from_teams(team_t t) {
  round_t R = 0;
  t = (t << 1) - 1;
  while (t >>= 1) ++R;
  return R;
}

// meet_round
//       The round where two teams meet.
round_t meet_round(const team_t i, const team_t j) {
  round_t R = 0;
  team_t v = i ^ j;
  while (v) {
    ++R;
    v >>= 1;
  }
  return R;
}

/*********************************************************************/
/* Worth functions
/*********************************************************************/

//
// built in worth_functions
//
int worth_exp_array[] = {0,1,2,4,8,16,32,64,128,256,512,1024,2048,
			       4096,8192,16384,32768,65536};

int worth_ESPN_array[] = {0,10,20,40,80,120,160};

worth_function_t worth_exp = worth_exp_array;
worth_function_t worth_ESPN = worth_ESPN_array;

//
// the global worth function
//
worth_function_t worth_function = worth_exp;

//
// worth_function_set
//    set a worth_function_t to be used as the worth function for all
//    future calculations
//
void worth_function_set(int *w) {
  worth_function = w;
}

//
// worth_max
//    returns the score that perfect picks would be worth
//
int worth_max(round_t rounds) {
  int tot = 0;
  for (round_t k=1; k<=rounds; k++)
    tot += (1 << (rounds-k))*worth(k);
  return tot;
}


int GCD(int a, int b)
{
  while( 1 )
    {
      a = a % b;
      if( a == 0 )
	return b;
      b = b % a;
      if( b == 0 )
	return a;
    }
}

//
// worth_gcd
//    returns the smallest difference between two possible scores
//
int worth_gcd(round_t rounds) {
  int g = worth(1);
  for (round_t k=2; k<=rounds; k++)
    g = GCD(g,worth(k));
  return g;
}



