/*
 * TourneyBasics.h
 *
 *   Fundamental definitions for working with tournaments.
 *   Includes basic types and worth functions.
 *
 * Copyright (C) 2005 Bryan Clair
 *
 * This file is part of CLOP.
 *
 * CLOP is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * CLOP is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with CLOP; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *
 */

#ifndef _TOURNEYBASICS_H_
#define _TOURNEYBASICS_H_


//
// Don't use more rounds than MAX_ROUNDS.
//    The limiting factor is # of bits in an unsigned long,
//    since indices are often calculated by bitshifting.
//    The value seemed small enough to be safe, but probably
//    could be made much larger.  If you care.
//
const int MAX_ROUNDS = 14;

/*********************************************************************/
/* Team and Round manipulations
/*********************************************************************/

typedef unsigned int team_t;
typedef unsigned int round_t;

// rounds_from_teams
//       Calculate # of rounds needed for given # of teams
round_t rounds_from_teams(team_t t);

// teams_from_rounds
//       Calculate # of teams in tournament with given # of rounds
inline team_t teams_from_rounds(round_t r) { return (1 << r); }

// round_mask
//       Bitmask for round r
//       Examples: round 0: 111...111
//                 round 1: 111...110
//                 round 2: 111...100
//       i and j share the same round r game if
//            i & round_mask(r) == j & round_mask(r)
inline team_t round_mask(const round_t r) { return ((~0) << r); }

// dependent
//       true if (i wins round k) and (j wins round l) are dependent events
//
inline bool dependent(const team_t i, const round_t k,
		      const team_t j, const round_t l) {
  return ((k > l) ?
	  ( (i & round_mask(k)) == (j & round_mask(k)) ) :
	  ( (i & round_mask(l)) == (j & round_mask(l)) ));
}
      
// meet_round
//       The round where two teams meet.
round_t meet_round(const team_t i, const team_t j);

// FOR_EACH_OPPONENT
//       WARNING:  only use for r > 0
//       A for loop that runs a variable opp of type team_t over all
//       possible round r opponents of team i.
#define FOR_EACH_OPPONENT(opp,i,r)\
   for (team_t opp = (i & round_mask(r-1)) ^ (1 << (r-1));\
	(opp ^ i) & (1 << (r-1)); opp++)

// FOR_EACH_TEAM_IN_REGION
//       A for loop that runs a variable opp of type team_t over all
//       possible participants in the round r game that team i is 
//       bracketed for.
#define FOR_EACH_TEAM_IN_REGION(opp,i,r)\
   for (team_t opp = (i & round_mask(r)), _TBFETIRend = opp + (1 << r);\
	opp < _TBFETIRend; opp++)

/*********************************************************************/
/* Worth functions
/*********************************************************************/

//
// worth_function_t
//
typedef int* worth_function_t;

//
// the global worth function - don't use this directly.
//
extern worth_function_t worth_function;

//
// built in worth_functions
//
extern worth_function_t worth_exp;  //  1,2,4,8,...   This is the default
extern worth_function_t worth_ESPN; //  10,20,40,80,120,160

//
// worth_function_set
//    set a worth function to be used as the worth function for all
//    future calculations
//
void worth_function_set(worth_function_t);

//
// worth
//    returns the worth of a correct pick in a given round,
//    using the current worth function.
//
inline int worth(const round_t round) { return worth_function[round]; }

//
// worth_max
//    returns the score that perfect picks would be worth
//
int worth_max(round_t rounds);

//
// worth_gcd
//    returns the smallest difference between two possible scores
//
int worth_gcd(round_t rounds);

#endif
