/*
 * TourneyFile.C
 *
 *   Read/write data in a tournament file.
 *
 * Copyright (C) 2005 Bryan Clair
 *
 * This file is part of CLOP.
 *
 * CLOP is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * CLOP is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with CLOP; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *
 */

#include <iostream>
#include <istream>
#include <fstream>
#include <string>
#include <sstream>
#include <stdio.h>
#include <stdexcept>

using namespace std;

#include "TourneyPicks.h"
#include "TourneyPack.h"
#include "TourneyPool.h"
#include "TourneyFile.h"

/*****************/
//
// TP_read_error
// exception class
//
TP_read_error::TP_read_error(string s) {
  err = "Tournament pool read error";
  if (s != "") err = err + ": " + s;
}

void TP_read_error::display() const {
  cerr << err << endl;
}

/*****************/
//
// Utility routines
//

//
//   despace - remove lead & trailing spaces from string
//
string despace(const string &s) {
  static const string whitespace=" \t\n";
  return s.substr
    (s.find_first_not_of(whitespace),
     s.find_last_not_of(whitespace) - s.find_first_not_of(whitespace) + 1);
}

//
//   stripquotes - remove lead & trailing " chars from string
//
string stripquotes(const string &s) {
  if ((s[0] == '"') && (s[s.length()-1] == '"'))
    return s.substr(1,s.length()-2);
  else return s;
}

/*****************/
//
// Display routines
//

//
// read_depth
// write_depth
//
//    read and write picks in depth-format, a 64 entry list of numbers
//    from 0..rounds, one for each team, indicating the round that team
//    reached in the picks.
//    read_depth returns false for failure.
//
void write_depth(const Picks& X, std::ostream &f) {
  team_t t;
  round_t r;
  int g;

  for (t = 0; t < X.teams(); t++) {
    r = 1;
    g = t >> 1;
    while ((r <= X.rounds()) && (X.winner(r,g) == t)) {
      g = g >> 1;
      ++r;
    }
    if (t > 0) f << ' ';
    f << r;
  }
}

void read_depth(Picks &X, std::istream &f) {
  team_t t;
  round_t r,depth;
  int g;

  for (t = 0; t < X.teams(); t++) {
    f >> depth;
    if (!f || (depth < 1) || (depth > X.rounds()+1))
      throw TP_read_error("depth out of range");
    g = t >> 1;
    r = 1;
    while (r < depth) {
      X.setwinner(r,g,t);
      g = g >> 1;
      ++r;
    }
  }
}  

/*****************/
//
// Read tournament data and name files
//

// read_probability
//   read one float and throw an exception if the read fails
//   or it's not in the range [0,1]
double read_probability(istream &f) {
  double x;
  f >> x;
  if (!f)
    throw TP_read_error("probability data not found");
  if (x < 0 || x > 1)
    throw TP_read_error("probability out of range");
  return x;
}

// read_new_winround
//    read a winround format array of probability data from a stream
//    and build a ProbSuite from it
ProbSuite *read_new_winround(istream& f, team_t teams) {
  round_t rounds = rounds_from_teams(teams);
  string header;

  f >> header;
  if (header != "solo")
    throw TP_read_error("solo header not found");

  double solo[teams * (rounds+1)];
  
  for (int n = 0; n < teams * (rounds+1); n++) {
    solo[n] = read_probability(f);
  }

  f >> header;
  if (header == "pair") {
    double pair[teams * (rounds+1) * teams * (rounds+1)];
    
    for (int n = 0; n < teams * teams * (rounds+1) * (rounds+1); n++) {
      pair[n] = read_probability(f);
    }
    
    Winround wdata(teams,solo,pair);
    
    return new ProbSuite(wdata);

  } else {
    std::cerr << "WARNING: proceeding without pair data\n";
    Winround wdata(teams,solo);
    return new ProbSuite(wdata);
  } 
}

// read_new_h2h
//    read an n x n array of probability data from a stream
//    and build a ProbSuite from it
ProbSuite *read_new_h2h(istream& f, team_t teams) {
  double data[teams * teams];
  for (int n = 0; n < teams*teams; n++) {
      data[n] = read_probability(f);
  }
  return new ProbSuite(teams,data);
}

// read_numteams
//    read the # of teams (and error check it)
team_t read_numteams(istream&f) {
  team_t T;
  f >> T;
  if ((!f) || (T == 0) || (rounds_from_teams(T) > MAX_ROUNDS) ||
      (T != teams_from_rounds(rounds_from_teams(T))))
    throw TP_read_error("bad # of teams");
  return T;
}

// read_rest_of_line
//    read the rest of the current line (with getline),
//    strip extra whitespace from beginning and end, and strip quotes.
string read_rest_of_line(istream& f) {
  string s;
  if (!getline(f,s))
    throw TP_read_error("reading line");
  try {
    return stripquotes(despace(s));
  } catch (out_of_range) {
    return "";
  }
}

// read_probfile
//     Read a ProbSuite from an h2h or winround type file
ProbSuite *read_probfile(istream& f) {
  ProbSuite *P;
  string element, title;
  
  f >> element;
  team_t numt = read_numteams(f);
  title = read_rest_of_line(f);

  if (element == "h2h") P = read_new_h2h(f,numt);
  else if (element == "winround") P = read_new_winround(f,numt);
  else throw TP_read_error("unrecognized data file type");

  P->set_title(title);

  return P;
}

// read_namesfile
//     Read teamnames from a file
string read_namesfile(istream& f, Names& teamnames)
{
  string element;
  f >> element;
  if (element != "names") 
    throw(TP_read_error("not a tournament names file"));
  
  team_t numt = read_numteams(f);
  string title = read_rest_of_line(f);
  
  teamnames.clear();
  for (team_t i = 0; i < numt; i++) {
    teamnames.push_back(read_rest_of_line(f));
  }

  return title;
}

// read_pool
//     Create a new pool from three files, passed as an array of filenames.
//     The first file is the names data, then the actual, then perceived.
//     Throws TP_read_error if unsuccessful
//
TourneyPool *read_pool(char *filename[]) {

  ifstream nf(filename[0]);
  if (!nf) throw TP_read_error("couldn't open names file");

  Names teamnames;
  string title = read_namesfile(nf, teamnames);

  ifstream af(filename[1]);
  if (!af) throw TP_read_error("couldn't open actual file");
  ProbSuite *a = read_probfile(af);

  ifstream pf(filename[2]);
  if (!pf) throw TP_read_error("couldn't open perceived file");
  ProbSuite *p = read_probfile(pf);

  TourneyPool *pool = new TourneyPool(*a,*p);
  pool->set_names(teamnames);
  pool->set_title(title);

  delete a;
  delete p;

  return pool;
}
