/*
 * TourneyFile.h
 *
 *   Tournament data I/O, including file input.
 *
 * Copyright (C) 2005 Bryan Clair
 *
 * This file is part of CLOP.
 *
 * CLOP is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * CLOP is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with CLOP; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *
 */

#ifndef TOURNEYFILE_H
#define TOURNEYFILE_H

#include <istream>
#include <ostream>
#include <string>
#include <vector>

#include "TourneyPack.h"
#include "TourneyNames.h"

class TourneyPool;

//    read and write picks in depth-format, a T entry list of numbers
//    from 1..R+1, one for each team, indicating the round that team
//    reached in the picks.
//    read_depth throws TP_read_error on failure
//
void write_depth(const Picks&, std::ostream& = std::cout);
void read_depth(Picks&, std::istream&);

/*********************************************************************/
/*  Tournament Data File Format                                      */
/*********************************************************************/
/*
 * Tournament team names are stored in a file as follows:
 *    names <N> <optional comment to end of line>
 *    <teamname1>
 *    <teamname2>
 *       ...
 *    <teamnameN>
 *
 * Tournament data is stored in files of two types:
 *    h2h
 *    winround
 *
 * Both types of data file have a header followed by data.
 *
 * The header consists of a tag, the number of teams, and a description.
 * The description is everything from the number of teams to the end of line.
 *
 * h2h files:
 *     Tag: "h2h"
 *     Data is an array of T*T floats, in order:
 *         P(0 beats 0) P(0 beats 1) .. P(0 beats T-1)
 *               ...
 *         P(T-1 beats 0) ..            P(T-1 beats T-1)
 *     The data is redundant since P(i beats j) = 1 - P(j beats i).
 *     Values for P(x beats x) are required but ignored. 
 *     
 * winround files:
 *     Tag: "winround"
 *     Data is the keyword "solo" followed by a wsolo array of floats,
 *     followed by the keyword "pair" followed by a wpair array of floats.
 *    
 *     The wsolo array is size (T * (R+1)), in order:
 *        P(0->0) P(0->1) ... P(0->R) P(1->0) ... P((T-1)->R)
 *
 *     The wpair array is, (T * T * (R+1) * (R+1)), in order:
 *        P(0->0 & 0->0) P(0->0 & 0->1) .. P(0->0 & 0->R)
 *        P(0->1 & 0->0) ..                P(0->1 & 0->R)
 *         ...
 *        P(0->R & 0->0) ..                P(0->R & 0->R)
 *        P(0->0 & 1->0) P(0->0 & 1->1) .. P(0->0 & 1->R)
 *         ...
 *        P(0->R & 1->0) ..                P(0->R & 1->R)
 *         ...
 *        P(0->R & (T-1)->0) ..        P(0->R & (T-1)->R)
 *        P(1->0 & 0->0) ...
 *                                 P((T-1)->R & (T-1)->R)
 *
 *********************************************************************/

// read_probfile
//     Allocate and read a new ProbSuite from an h2h or winround type file
ProbSuite *read_probfile(std::istream&);

// write_probfile
//     Write a ProbSuite to an h2h type file, or if h2h is
//     unavailable, write to a winround type file.
// UNIMPLEMENTED
void write_probfile(std::ostream&, const ProbSuite&);

// read_namefile
//     Read teamnames from a file, return tournament title
std::string read_namesfile(std::istream&, Names&);

// write_namefile
// UNIMPLEMENTED
void write_namefile(std::ostream&, const Names&);


// read_pool
//     Create a new pool from three files, passed as an array of filenames.
//     The first file is the names data, then the actual, then perceived.
//     Throws TP_read_error if unsuccessful
TourneyPool *read_pool(char *filename[]);

//
// TP_read_error
//    class for format errors when reading pool datafiles
//
class TP_read_error {
 public:
  TP_read_error(std::string s="");
  void display() const;
 private:
  std::string err;
};

#endif
