/*
 * TourneyH2H.C
 *
 *   Manages one matrix of head-to-head probabilities for a tournament.
 *
 * Copyright (C) 2005 Bryan Clair
 *
 * This file is part of CLOP.
 *
 * CLOP is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * CLOP is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with CLOP; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *
 */

#include <stdio.h>
#include <assert.h>

using namespace std;

#include "TourneyBasics.h"
#include "TourneyH2H.h"
#include "TourneyWinround.h"

HeadToHead::HeadToHead() {
  numt = 0;
  data = NULL;
}

HeadToHead::~HeadToHead() {
  if (data != NULL) delete[] data;
}

HeadToHead::HeadToHead(const HeadToHead& h) {
  numt = h.numt;
  data = new double[numt*numt];
  for (int i = 0; i < numt*numt; i++) data[i] = h.data[i];
}

HeadToHead& HeadToHead::operator=(const HeadToHead& h) {
  if (this != &h) {
    numt = h.numt;
    if (data != NULL) delete[] data;
    data = new double[numt*numt];
    for (int i = 0; i < numt*numt; i++) data[i] = h.data[i];
  }
  return *this;
}

// constructor
//        Pass # of teams and 2d array of head-to-head data
HeadToHead::HeadToHead(team_t T, double *P) {
  numt = T;
  data = new double[numt*numt];
  for (int i = 0; i < numt*numt; i++) data[i] = P[i];
}

// constructor
//        From Winround data
HeadToHead::HeadToHead(const Winround& w) {
  round_t r;

  numt = w.teams();
  data = new double[numt*numt];

  for (team_t i=0; i < numt; i++) {
    for (team_t j=0; j < i; j++) {
      r = meet_round(i,j);
      if (w(i,r,j,r-1) + w(i,r-1,j,r) == 0) {
	data[i*numt+j] = data[j*numt+i] = 0.5;
      } else {
	data[i*numt+j] = w(i,r,j,r-1)/(w(i,r,j,r-1) + w(i,r-1,j,r));
	data[j*numt+i] = w(i,r-1,j,r)/(w(i,r,j,r-1) + w(i,r-1,j,r));
      }
    }
  }
}

// operator()
//        Return head-to-head probability that team t1 beats team t2.
double HeadToHead::operator() (team_t t1, team_t t2) const {
  assert(0 <= t1 && t1 < numt);
  assert(0 <= t2 && t2 < numt);
  assert(t1 != t2);
  return data[t1*numt+t2];
}

// dump
//        Dump data to stdout in a stream of floats.
void HeadToHead::dump() const {
  team_t i,j;
  for (i = 0; i < teams(); i++) {
    for (j = 0; j < teams(); j++) {
      if (i==j) printf("0.500 ");
      else printf("%.3f ",(*this)(i,j));
    }
    printf("\n");
  }
}

// display
//        Display all data on stdout, nicely formatted, use teamnames if given.
void HeadToHead::display(char *teamnames[]) const {
  team_t i,j;

  printf("W \\ L");
  for (i = 0; i < teams(); i++) {
    if (teamnames)
      printf(" [%s] ",teamnames[i]);
    else
      printf(" [%2d] ",i);
  }
  printf("\n");
  for (i = 0; i < teams(); i++) {
    if (teamnames != NULL)
      printf("[%s] ",teamnames[i]);
    else
      printf("[%2d] ",i);
    for (j = 0; j < teams(); j++) {
      if (i==j) printf("  -   ");
      else {
	printf("%.3f ",(*this)(i,j));
      }
    }
    printf("\n");
  }
}
