/*
 * TourneyPack.C
 *
 *   Package for working with elimination tournament pools.
 *
 * Copyright (C) 2005 Bryan Clair
 *
 * This file is part of CLOP.
 *
 * CLOP is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * CLOP is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with CLOP; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *
 */

#include <iostream>
#include <cstdio>
#include <iomanip>
#include <cmath>
#include <cassert>

using namespace std;

#include "PoolMath.h"
#include "TourneyBasics.h"
#include "TourneyPicks.h"
#include "TourneyH2H.h"
#include "TourneyWinround.h"
#include "TourneyPack.h"

/*********************************************************************/
/*   ProbSuite basic interface                                       */
/*********************************************************************/
// Default constructor
//
ProbSuite::ProbSuite()
{
  numt = 0;
  numr = 0;
}

// constructor
//        With array of head-to-head data (teams x teams)
ProbSuite::ProbSuite(team_t T, double *data)
  : h2h(T,data),
    wins(h2h)
{
  numt = T;
  numr = rounds_from_teams(T);
}

// constructor
//        With head-to-head data
ProbSuite::ProbSuite(const HeadToHead& h)
  : h2h(h),
    wins(h)
{
  numt = h.teams();
  numr = rounds_from_teams(numt);
}

// constructor
//        With winround data
ProbSuite::ProbSuite(const Winround& w)
  : wins(w),
    h2h(w)
{
  numt = w.teams();
  numr = rounds_from_teams(numt);
}

//
// Copy constructor
//
ProbSuite::ProbSuite(const ProbSuite& A)
 : numt(A.numt), numr(A.numr),
   h2h(A.h2h), wins(A.wins), mytitle(A.mytitle)
{
}

//
// Assignment operator
//
ProbSuite& ProbSuite::operator=(const ProbSuite& A) {
  if (this != &A) {
    numt = A.numt;
    numr = A.numr;
    h2h = A.h2h;
    wins = A.wins;
    mytitle = A.mytitle;
  }
  return *this;
}

//
// Destructor
//
ProbSuite::~ProbSuite() {
}

// display
//        Dump most useful data to cout
void ProbSuite::display(char *teamnames[], bool all) {
  printf("--- ProbSuite %s begins ---\n",mytitle.c_str());

  printf("HEAD TO HEAD PROBABILITIES:\n");
  h2h.display(teamnames);

  printf("PROBABILITIES OF WINNING ROUNDS:\n");
  if (all) wins.display_all(teamnames);
  else wins.display(teamnames);

  printf("--- ProbSuite %s ends ---\n", mytitle.c_str());
}

/*********************************************************************/
/*   ProbSuite picks creation routines                               */
/*********************************************************************/

// favorites
//    pick the favorite in every game
//
Picks ProbSuite::favorites() const {
  Picks p(rounds());
  int i, tot = 0;
  round_t k;
  team_t t1,t2;

  for (k = 1; k <= rounds(); k++)
    for (i = 0; i < (1 << (rounds()-k)); i++) {
      t1 = p.winner(k-1,2*i);
      t2 = p.winner(k-1,2*i+1);
      p.setwinner(k,i, (h2h(t1,t2) >= .5) ? t1 : t2);
    }

  return p;
}

// random_picks
//    pick randomly with h2h probabilities
//
Picks ProbSuite::random_picks() const {
  Picks p(rounds());

  int i, tot = 0;
  round_t k;
  team_t t1,t2;

  for (k = 1; k <= rounds(); k++)
    for (i = 0; i < (1 << (rounds()-k)); i++) {
      t1 = p.winner(k-1,2*i);
      t2 = p.winner(k-1,2*i+1);
      p.setwinner(k,i, (rand_prob() < h2h(t1,t2)) ? t1 : t2);
    }

  return p;
}

//
// most_likely
//
// finds the set of picks most likely to occur (100% correct).
//
// i.e. the maximum over all sets of picks of the product of
// h2h(team1,team2), taken over games that occur in the set of picks.
//
Picks ProbSuite::most_likely(double *optimum) const {
  team_t T = teams();
  team_t R = rounds();

  // val[i][k] is the optimial value for team i winning round k.
  double val[T][R+1];

  // beaten_by[i][k] is the team that team i beats to win round k
  // in order to optimize val[i][k]
  team_t beaten_by[T][R+1];

  // Calculate val and beaten_by

  for (team_t i=0; i < T; i++)
    val[i][0] = 1;

  for (round_t r=1; r <= R; r++) {
    for (team_t i=0; i < T; i++) {
      val[i][r] = -1;
      // j runs through the sub-bracket opposite team i for round r.
      FOR_EACH_OPPONENT(j,i,r) {
	double v = val[i][r-1] * val[j][r-1] * h2h(i,j);
	if (v > val[i][r]) {
	  val[i][r] = v;
	  beaten_by[i][r] = j;
	}
      }
    }
  }

  // Find the winner

  team_t winner = 0;
  for (team_t i=0; i < T; i++) {
    if (val[i][R] > val[winner][R]) winner = i;
  }

  if (optimum != NULL) *optimum = val[winner][R];

  // Fill in picks, starting with winner and working
  // backwards, using the fact that you know who a team
  // beat to reach a given point.

  Picks best_picks(R);
  best_picks.setwinner(R,0,winner);

  for (round_t r = R; r > 1; r--) {
    for (int game = 0; game < (1 << (R-r)); game++) {
      winner = best_picks.winner(r,game);
      if (winner > beaten_by[winner][r]) {
	best_picks.setwinner(r-1,2*game,beaten_by[winner][r]);
	best_picks.setwinner(r-1,2*game+1,winner);
      } else {
	best_picks.setwinner(r-1,2*game,winner);
	best_picks.setwinner(r-1,2*game+1,beaten_by[winner][r]);
      }
    }
  }

  return best_picks;
}

// max_expected_score
//    finds the set of picks with the highest expected score 
//    (return expected value if non-NULL)
//
Picks ProbSuite::max_expected_score(double *optimum) const {
  team_t T = teams();
  team_t R = rounds();

  // val[i][k] is the optimial value for team i winning round k.
  double val[T][R+1];

  // beaten_by[i][k] is the team that team i beats to win round k
  // in order to optimize val[i][k]
  team_t beaten_by[T][R+1];

  // Calculate val and beaten_by

  for (team_t i=0; i < T; i++)
    val[i][0] = 0;

  for (round_t r=1; r <= R; r++) {
    for (team_t i=0; i < T; i++) {
      val[i][r] = -1;
      // j runs through the sub-bracket opposite team i for round r.
      FOR_EACH_OPPONENT(j,i,r) {
	double v = val[i][r-1] + val[j][r-1] + worth(r) * wins(i,r);
	if (v > val[i][r]) {
	  val[i][r] = v;
	  beaten_by[i][r] = j;
	}
      }
    }
  }

  // Find the winner

  team_t winner = 0;
  for (team_t i=0; i < T; i++) {
    if (val[i][R] > val[winner][R]) winner = i;
  }

  if (optimum != NULL) *optimum = val[winner][R];

  // Fill in picks, starting with winner and working
  // backwards, using the fact that you know who a team
  // beat to reach a given point.

  Picks best_picks(R);
  best_picks.setwinner(R,0,winner);

  for (round_t r = R; r > 1; r--) {
    for (int game = 0; game < (1 << (R-r)); game++) {
      winner = best_picks.winner(r,game);
      if (winner > beaten_by[winner][r]) {
	best_picks.setwinner(r-1,2*game,beaten_by[winner][r]);
	best_picks.setwinner(r-1,2*game+1,winner);
      } else {
	best_picks.setwinner(r-1,2*game,winner);
	best_picks.setwinner(r-1,2*game+1,beaten_by[winner][r]);
      }
    }
  }

  return best_picks;
}

//
// best_for_large_pool
//
//   Choose picks that maximize expected value for a pool
//   with a very large ( >> 2^teams) number of opponents.
//   To do this, find the maximum over all sets of picks of
//   the product of (a/p) for each game in picks.
//
Picks best_for_large_pool(const ProbSuite& a, const ProbSuite& p, double *optimum)
{
  assert(a.rounds() == p.rounds());

  team_t T = a.teams();
  team_t R = a.rounds();

  // val[i][k] is the optimial value for team i winning round k.
  double val[T][R+1];

  // beaten_by[i][k] is the team that team i beats to win round k
  // in order to optimize val[i][k]
  team_t beaten_by[T][R+1];

  // Calculate val and beaten_by

  for (team_t i=0; i < T; i++)
    val[i][0] = 1;

  for (round_t r=1; r <= R; r++) {
    for (team_t i=0; i < T; i++) {
      val[i][r] = -1;
      // j runs through the sub-bracket opposite team i for round r.
      FOR_EACH_OPPONENT(j,i,r) {
	double v = val[i][r-1] * val[j][r-1] * a.h2h(i,j) / p.h2h(i,j);
	if (v > val[i][r]) {
	  val[i][r] = v;
	  beaten_by[i][r] = j;
	}
      }
    }
  }

  // Find the winner

  team_t winner = 0;
  for (team_t i=0; i < T; i++) {
    if (val[i][R] > val[winner][R]) winner = i;
  }

  if (optimum != NULL) *optimum = val[winner][R];

  // Fill in picks, starting with winner and working
  // backwards, using the fact that you know who a team
  // beat to reach a given point.

  Picks best_picks(R);
  best_picks.setwinner(R,0,winner);

  for (round_t r = R; r > 1; r--) {
    for (int game = 0; game < (1 << (R-r)); game++) {
      winner = best_picks.winner(r,game);
      if (winner > beaten_by[winner][r]) {
	best_picks.setwinner(r-1,2*game,beaten_by[winner][r]);
	best_picks.setwinner(r-1,2*game+1,winner);
      } else {
	best_picks.setwinner(r-1,2*game,winner);
	best_picks.setwinner(r-1,2*game+1,beaten_by[winner][r]);
      }
    }
  }

  return best_picks;
}

/*********************************************************************/
/* Routines to calculate statistics of tourneys, Picks & ProbSuites  */
/*********************************************************************/

//
// score
//   Of a pair of picks (points for picks which agree)
//
int score(const Picks& p1, const Picks& p2) {

  int i, tot = 0;
  round_t R = p1.rounds(), k;

  for (k = 1; k <= R; k++)
    for (i = 0; i < (1 << (R-k)); i++)
      if (p1.winner(k,i)==p2.winner(k,i)) tot += worth(k);

  return tot;
}

//
//  mean_for_random_picks
//    Mean score if picking teams at random (50-50).
//    Depends only on number of rounds in tournament and on
//    the worth function.
//
double mean_for_random_picks(round_t rounds) {
  double tot = 0;
  // Calculate 2^rounds * sum{k=1..rounds}(worth(k) / 4^k)
  for (round_t k = 1; k <= rounds; k++)
    tot += (worth(k) / (double)(1 << k)) * (1 << rounds-k);
  return tot;
}


// mean
//    score if you choose picks using A, picks using B and compare.
//    (symmetric in A and B):
double mean(const ProbSuite& A, const ProbSuite& B)
{
  assert (A.rounds()==B.rounds());
  
  double Ex=0, Exk;
  team_t i;
  round_t k;

  for (k = 1; k <= A.rounds(); k++) {
    Exk = 0;
    for (i = 0; i< A.teams(); i++)
      Exk += A.wins(i,k) * B.wins(i,k);
    Ex += worth(k) * Exk;
  }
  return Ex;
}

// probability
//    that a set of picks occurs:
double probability(const ProbSuite& A, const Picks& picks)
{
  assert(A.rounds() == picks.rounds());

  round_t k, R = picks.rounds();
  int i;
  team_t winner, loser;
  double p=1;

  for (k = R; k > 0; k--) {
    for (i = 0; i < (1 << (R-k)); i++) {
      winner = picks.winner(k,i);
      loser = picks.winner(k-1,2*i);
      if (winner == loser) loser = picks.winner(k-1,2*i+1);
      p *= A.h2h(winner,loser);
    }
  }

  return p;
}

// Mean score for given picks, actual probs.
//
double mean(const ProbSuite& A, const Picks& picks) {
  assert(A.rounds() == picks.rounds());

  round_t k, R = picks.rounds();
  int i;
  double tot = 0;

  for (k = 1; k <= R; k++)
    for (i = 0; i < (1 << (R-k)); i++)
      tot += worth(k) * A.wins(picks.winner(k,i),k);
  
  return tot;
}

// variance
//    of the score if you choose picks using A and compare to picks.
//    Tried to make this fast.
//
double variance(const ProbSuite& A, const Picks& picks) {
  assert(A.rounds() == picks.rounds());

  int g,h;
  round_t k,l, R = A.rounds();
  team_t i,j;
  double v = 0, s;

  // (k,g) loops through all games in the picks
  for (k = 1; k <= R; k++) {
    for (g = 0; g < (1 << (R - k)); g++) {
      i = picks.winner(k,g);
      
      // game (i,k) depends on game (i,k)
      s = worth(k)*(A.wins(i,k)*(1-A.wins(i,k)));

      // (l,h) loops through all games dependent on (k,g) with l > k
      h = g/2; l = k+1;
      while (l <= R) {
	j = picks.winner(l,h);
	s += 2*worth(l)*(A.wins(i,k,j,l) - A.wins(i,k)*A.wins(j,l));
	l++;
	h /= 2;
      };

      v += worth(k)*s;
    }
  }

  return v;
}

// variance
//    of the score if you choose picks using A and B and compare.
//    (symmetric in A and B)
double variance(const ProbSuite& A, const ProbSuite& B) {
  assert(A.rounds() == B.rounds());

  team_t i,j;
  round_t k,l;
  double v = 0,s;

  // room for speed improvement:
  //    just sum over the dependent games, and only
  //    those with k <= l.
  //
  for (k = 1; k <= A.rounds(); k++) {
    for (l = 1; l <= A.rounds(); l++) {
      s = 0;
      for (i = 0; i < A.teams(); i++) {
	for (j = 0; j < A.teams(); j++) {
	  if (dependent(i,k,j,l)) {
	    s += A.wins(i,k,j,l)*B.wins(i,k,j,l)
	      - A.wins(i,k)*A.wins(j,l)*B.wins(i,k)*B.wins(j,l);
	  }
	}
      }
      v += s * worth(k)*worth(l);
    }
  }
  return v;
}

// covariance
//    of the scores of two players, making given picks1 and picks2.
//
double covariance(const ProbSuite& A, const Picks& picks1, const Picks& picks2) {
  assert(A.rounds() == picks1.rounds());
  assert(A.rounds() == picks2.rounds());

  int g,h;
  round_t k,l, R = A.rounds();
  team_t i,j;
  double v = 0, s;

  // (k,g) loops through all games in picks1
  for (k = 1; k <= R; k++) {
    for (g = 0; g < (1 << (R - k)); g++) {
      i = picks1.winner(k,g);

      // 
      // Now compute the contribution to the sum that contains picks1's
      // bet on game (i,k).
      // Each term is worth(k)*worth(l)*picks1(i->k)*picks2(j->l)*
      //             [A(i->k && j->l) - A(i->k)*A(j->l)]
      s = 0;

      // (l,h) loops through all games in picks2
      for (l = 1; l <= R; l++) {
	for (h = 0; h < (1 << (R - l)); h++) {
	  j = picks2.winner(l,h);
	  if (dependent(i,k,j,l)) {
	    s += worth(l)*(A.wins(i,k,j,l) - A.wins(i,k)*A.wins(j,l));
	  }
	}
      }

      v += worth(k)*s;
    }
  }
  return v;
}

// covariance
//    of the scores of two players, one using randomized strategy P,
//    and one making the given set of picks.
//
double covariance(const ProbSuite& A, const ProbSuite& P, const Picks& picks) {
  assert(A.rounds() == picks.rounds());
  assert(A.rounds() == P.rounds());

  int g;
  round_t r,s, R = A.rounds();
  team_t i,T = A.teams();
  double v = 0, sum;

  // (r,g) loops through all games in the picks
  for (r = 1; r <= R; r++) {
    for (g = 0; g < (1 << (R - r)); g++) {
      i = picks.winner(r,g);

      // Now compute the contribution to the sum that contains our
      // bet on game (i,r).

      // Run (j,s) over all events (j->s) for which
      // (i->r) and (j->s) are dependent.

      // Each term contributes
      //     worth(r)*worth(s)*pick(i->r)*P(j->s)*
      //             [A(i->r && j->s) - A(i->r)*A(j->s)]

      sum = 0;

      // Every pair (j,s) that feeds into (i,r) is a dependency
      FOR_EACH_TEAM_IN_REGION(j,i,r) {
	for (s = 1; s <= r; s++) {
	  sum += worth(s)*P.wins(j,s)*
	    (A.wins(i,r,j,s) - A.wins(i,r)*A.wins(j,s));
	}
      }

      // Every game that (i,r) feeds into is a dependency
      for (s = r+1; s <= R; s++) {
	FOR_EACH_TEAM_IN_REGION(j,i,s) {
	  sum += worth(s)*P.wins(j,s)*
	    (A.wins(i,r,j,s) - A.wins(i,r)*A.wins(j,s));
	}
      }

      // Might be worth precomputing a table of the A part of the term
      // for every i,k,j,l
      
      v += worth(r)*sum;
    }
  }
  return v;
}

// covariance
//    of the scores of two players using randomized strategy P1 and P2
double covariance(const ProbSuite& A, const ProbSuite& P1, const ProbSuite& P2) {
  assert(A.rounds() == P1.rounds());
  assert(A.rounds() == P2.rounds());

  team_t i,j;
  round_t k,l;
  double v = 0,s;

  // room for speed improvement:
  //    just sum over the dependent games, and only
  //    those with k <= l.
  //
  for (k = 1; k <= A.rounds(); k++) {
    for (l = 1; l <= A.rounds(); l++) {
      s = 0;
      for (i = 0; i < A.teams(); i++) {
	for (j = 0; j < A.teams(); j++) {
	  if (dependent(i,k,j,l)) {
	    s += (A.wins(i,k,j,l) - A.wins(i,k)*A.wins(j,l))
	         * P1.wins(i,k) * P2.wins(j,l);
	  }
	}
      }
      v += s * worth(k)*worth(l);
    }
  }
  return v;
}
