/*
 * TourneyPack.h
 *
 *   Package for working with elimination tournament pools.
 *
 * Copyright (C) 2005 Bryan Clair
 *
 * This file is part of CLOP.
 *
 * CLOP is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * CLOP is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with CLOP; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *
 */

#ifndef _TOURNEYPACK_H_
#define _TOURNEYPACK_H_

#include <string>

#include "TourneyBasics.h"
#include "TourneyPicks.h"
#include "TourneyH2H.h"
#include "TourneyWinround.h"

/*********************************************************************/
/*   ProbSuite                                                       */
/*********************************************************************/

class ProbSuite {
 public:
  //////////////////////////////////////////////////
  // Basic operations and constructors
  //////////////////////////////////////////////////

  ProbSuite();
  ~ProbSuite();                            // destructor
  ProbSuite(const ProbSuite&);             // copy constructor
  ProbSuite& operator=(const ProbSuite&);  // assignment

  // constructor
  //        With head-to-head data
  ProbSuite(const HeadToHead&);

  // constructor
  //        With winround data
  ProbSuite(const Winround&);

  // constructor
  //        With array of head-to-head data (teams x teams)
  ProbSuite(team_t, double *);

  //////////////////////////////////////////////////
  // Data access
  //////////////////////////////////////////////////

  // rounds
  // teams
  //        Return number of rounds/teams
  inline round_t rounds(void) const { return numr; }
  inline team_t teams(void) const { return numt; }

  // title stuff
  const std::string& title() const { return mytitle; }
  void set_title(const std::string& t) { mytitle = t; }

  // Probabilities
  const HeadToHead& head_to_head() { return h2h; }
  const Winround& winround() { return wins; }

  //////////////////////////////////////////////////
  // Display
  //////////////////////////////////////////////////

  // display
  //        Dump to cout
  //        If all is true, even show the winround pair data.
  void display(char *teamnames[] = NULL, bool all = false);
  
  //////////////////////////////////////////////////
  // Create Picks
  //////////////////////////////////////////////////

  // random_picks
  //        pick randomly with h2h probabilities
  Picks random_picks() const;

  // favorites
  //        pick the favorite in every game
  Picks favorites() const;

  // most_likely
  //        finds the set of picks most likely to occur (100% correct).
  //        (return probability of occurrence if non-NULL)
  Picks most_likely(double *probability = NULL) const;

  // max_expected_score
  //        finds the set of picks with the highest expected score 
  //        (return expected value if non-NULL)
  Picks max_expected_score(double *expected = NULL) const;

  // best_for_large_pool
  //        Find picks that maximize expected value for a pool
  //        with a very large ( >> 2^teams) number of opponents.
  //        Arguments are actual and perceived probability data.
  //        If return_on_bet != NULL, the expected return on a bet
  //        of 1 will be stored there.
  friend Picks best_for_large_pool(const ProbSuite& actual,
                                   const ProbSuite& perceived,
				   double *return_on_bet=NULL);

  //
  // Statistical calculations, detailed outside class declaration (below)
  //

  friend double probability(const ProbSuite&, const Picks&);
  friend double mean(const ProbSuite&, const Picks&);
  friend double mean(const ProbSuite&, const ProbSuite&); 
  friend double variance(const ProbSuite&, const Picks&);
  friend double variance(const ProbSuite&, const ProbSuite&);
  friend double covariance(const ProbSuite&, const Picks&, const Picks&);
  friend double covariance(const ProbSuite&, const ProbSuite&, const Picks&);
  friend double covariance(const ProbSuite&, const ProbSuite&, const ProbSuite&);

 private:
  round_t numr;
  team_t numt;
  std::string mytitle;
  HeadToHead h2h;
  Winround wins;
};

/*********************************************************************/
/*   Functions to calculate statistics                               */
/*********************************************************************/

///////////////////////////
// score
////////////////////////////

int score(const Picks&, const Picks&);
//    of a pair of picks (points for picks which agree)

////////////////////////////
// probability
////////////////////////////

double probability(const ProbSuite& A, const Picks& picks);
//    that a set of picks occurs (using the h2h data)

////////////////////////////
// mean
////////////////////////////

double mean(const ProbSuite& A, const Picks& picks);
//    Mean score if you choose picks using A and compare to picks.

double mean(const ProbSuite& A, const ProbSuite& P);
//    Mean score if you choose picks using A, picks using P and compare.
//    (symmetric in A and P)

double mean_for_random_picks(round_t rounds);
//    Mean score if picking teams at random (50-50).
//    Depends only on number of rounds in tournament and on
//    the worth function.

////////////////////////////
// variance
////////////////////////////

double variance(const ProbSuite& A, const Picks& picks);
//    of the score if you choose picks using A and compare to picks.

double variance(const ProbSuite& A, const ProbSuite& P);
//    of the score if you choose picks using A and P and compare.
//    (symmetric in A and P)

////////////////////////////
// covariance
////////////////////////////

double covariance(const ProbSuite& A, const Picks& picks1, const Picks& picks2);
//    of the scores of two players, making given picks1 and picks2.

double covariance(const ProbSuite& A, const ProbSuite& P, const Picks& picks);
//    of the scores of two players, one using randomized strategy P,
//    and one making the given set of picks.

double covariance(const ProbSuite& A, const ProbSuite& P1, const ProbSuite& P2);
//    of the scores of two players using randomized strategy P1 and P2

#endif
