/*
 * TourneyPicks.C
 *
 *   Package for working with elimination tournament picks.
 *
 * Copyright (C) 2005 Bryan Clair
 *
 * This file is part of CLOP.
 *
 * CLOP is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * CLOP is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with CLOP; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *
 */

#include <string>
#include <sstream>
#include <iostream>
#include <cassert>
#include <cstdlib>

using namespace std;

#include "TourneyBasics.h"
#include "TourneyPicks.h"

/*********************************************************************/
/*   Picks class
/*********************************************************************/

//
// picks are stored in a 1D array, with the unneccesary addition of the
// round 0 picks (i.e. the teams in the tournament).  Picks are stored
// as follows for an 8 team tourney where all lower # teams advance:
// 01234567 0246 04 0
// (spaces added for readability, not reality)
//

// Constructor
Picks::Picks(round_t R) {
  assert(R < MAX_ROUNDS);

  numr = R;
  numt = 1 << numr;
  picks = new team_t[2*numt];

  // Initialize to all low # seeds winning
  int p=0;
  for (round_t r = 0; r<=numr; r++) {
    for (team_t i = 0; i < numt; i += (1<<r)) {
      picks[p++] = i;
    }
  }
}

// Copy constructor
Picks::Picks(const Picks& P) {
  numr = P.numr;
  numt = P.numt;
  picks = new team_t[2*numt];
  for (int i=0; i<2*numt; i++) picks[i] = P.picks[i];
}

// Assignment operator
Picks& Picks::operator=(const Picks& P) {
  if (this != &P) {
    if (numt != P.numt) {
      numr = P.numr;
      numt = P.numt;
      delete[] picks;
      picks = new team_t[2*numt];
    }
    for (int i=0; i<2*numt; i++) picks[i] = P.picks[i];
  }
  return *this;
}

// Destructor
Picks::~Picks() {
  delete[] picks;
}

//
// Picks data access
//
team_t Picks::winner(round_t r, int game) const {
  assert(r>=0 && r <= numr);
  assert(game>=0 && (game < (1 << numr-r)));

  return picks[(((1 << r) - 1) << (numr - r + 1)) + game];
}

void Picks::setwinner(round_t r, int game, team_t team) {
  assert(r>0 && r <= numr);
  assert(game>=0 && (game < (1 << numr-r)));
  assert(team>=0 && team < numt);

  picks[(((1 << r) - 1) << (numr - r + 1)) + game] = team;
}


// teamtext
//    Return ascii text to display team.
//    Uses name from teamnames or number if teamnames == NULL
//    Restrict to width chars (if non zero)
//
string teamtext(const int num,
		const Names& teamnames,
		const int width = 0) {
  if (!teamnames.empty()) {
    return "[" + teamnames[num] + "]";
  } else {
    ostringstream s;
    s << '[';
    if (num < 10) s << ' ';
    s << num << ']';
    return s.str();
  }
}

// display Picks
//    in an ASCII art bracket format using team numbers or optional team names
void Picks::display(const Names& teamnames) const {
  int p = 0, game;
  int ct;  // # of matchups printed on current line

  for (round_t r = 0; r < rounds(); r++) {
    cout << "round " << r+1 << ":" << endl;
    ct = 0;
    game = 0;
    do {
      if (ct == 2) {
	ct = 0; cout << endl;
      } else if (ct != 0) {
	cout << ' ';
      }

      cout << teamtext(winner(r,game++),teamnames);
      cout << 'v';
      cout << teamtext(winner(r,game++),teamnames);

      ct++;

    } while (game < (1 << (rounds()-r)));
    cout << endl;
  }
  cout << "Champion: " << endl;
  cout << teamtext(winner(rounds(),0),teamnames) << endl;
}


//
// random_picks
//    Generate random (all games 50-50) picks
//
Picks random_picks(round_t rounds) {
  Picks p(rounds);

  int i, tot = 0;
  round_t k;
  team_t t1,t2;

  for (k = 1; k <= rounds; k++)
    for (i = 0; i < (1 << (rounds-k)); i++) {
      t1 = p.winner(k-1,2*i);
      t2 = p.winner(k-1,2*i+1);
      p.setwinner(k,i, ((random() % 2) == 0) ? t1 : t2);
    }

  return p;
}

// neighbor picks 
//   differ by the outcome of one game and all consequences -
//   the new winner of that game now goes as deep as the previous
//   winner.
//   There are numt-1 games, so numt-1 neighbors.
//   neighbor(0) flops the championship game,
//   neighbor(1), neighbor(2) flop the semifinals, and so on.
//
Picks Picks::neighbor(int i) const {
  assert(0 <= i && i < numt-1);
  // find the highest bit set in i+1, counting rounds as we go,
  // and ending up with a mask to find the game number.

  i++;
  round_t r = 1;
  int mask = 1 << (numr - 1);
  while (!(i & mask)) {
    r++;
    mask >>= 1;
  }

  i &= ~mask;

  // game to switch is round r, game i
  // find old winner (cursed) and new winner (blessed)
  team_t cursed = winner(r,i);
  team_t blessed = winner(r-1,i << 1);
  if (blessed == cursed) blessed = winner(r-1,(i << 1) + 1);

  Picks nbr(*this);
  do {
    nbr.setwinner(r,i,blessed);
    r++;
    i >>= 1;
  } while ((r <= numr) && (winner(r,i) == cursed));

  return nbr;
}

// near_neighbor picks differ in exactly one game
//    There are 2^(numr-1) = numt/2 near_neighbors.
//    near_neighbor(i) returns the set of picks in which winner(1,i)
//    loses one round earlier.
// ** This might be buggy for 4 team tournaments
Picks Picks::near_neighbor(int i) const {
  assert(0 <= i && i < numt/2);

  Picks np = *this;
  team_t a=0,b,doomed = winner(1,i);
  round_t r=1;
  
  // find 
  while ((r <= numr) && winner(r,i) == doomed) {
    b = 1-a;
    a = i & 1;
    r++;
    i = i/2;
  }
  
  np.setwinner(r-1,2*i+a,winner(r-2,4*i+2*a+b));
  return np;
}
