/*
 * TourneyPicks.h
 *
 *   Package for working with elimination tournament picks.
 *
 * Copyright (C) 2005 Bryan Clair
 *
 * This file is part of CLOP.
 *
 * CLOP is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * CLOP is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with CLOP; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *
 */

#ifndef _TOURNEYPICKS_H_
#define _TOURNEYPICKS_H_

#include "TourneyBasics.h"
#include "TourneyNames.h"

/*********************************************************************/
/*   Picks class                                                     */
/*********************************************************************/
/*
 *  Class holds one set of picks in a tournament.
 *
 */
class Picks {
 public:
  Picks(round_t rounds=0);
  Picks(const Picks&);
  Picks& operator=(const Picks&);
  ~Picks();

  round_t rounds(void) const { return numr; }
  team_t teams(void) const { return numt; }

  // get and set winners of games
  //    r = 0..rounds(), 0 <= game < 2^(rounds() - r)
  //    winner(0,x) is x
  //    setwinner(0,x) is illegal
  //
  team_t winner(round_t r, int game) const; 
  void setwinner(round_t r, int game, team_t i);

  // display Picks
  //    in an ASCII art bracket format using team numbers or optional team names
  void display(const Names& = Names(0)) const;

  // neighbor picks 
  //   differ by the outcome of one game and all consequences -
  //   the new winner of that game now goes as deep as the previous
  //   winner.
  //   There are teams-1 games, so teams-1 neighbors, numbered 0..teams-2.
  //   neighbor(0) flops the championship game,
  //   neighbor(1), neighbor(2) flop the semifinals, and so on.
  //
  team_t num_neighbors() const { return numt - 1; }
  Picks neighbor(int i) const;

  // near_neighbor picks differ in exactly one game
  //    There are 2^(rounds-1) = teams/2 near_neighbors.
  //    near_neighbor(i) returns the set of picks in which winner(1,i)
  //    loses one round earlier.
  team_t num_near_neighbors() const { return numt/2; }
  Picks near_neighbor(int i) const;

 private:
  round_t numr;
  team_t numt;
  team_t *picks;
};

//
// random_picks
//    Generate random (all games 50-50) picks
//
Picks random_picks(round_t rounds);

#endif
