/*
 * TourneyPool.C
 *
 *   Manage data in a football pool.
 *
 * Copyright (C) 2005 Bryan Clair
 *
 * This file is part of CLOP.
 *
 * CLOP is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * CLOP is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with CLOP; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *
 */

#include <istream>
#include <ostream>
#include <string>
#include <assert.h>

#include "TourneyPack.h"
#include "TourneyPool.h"
#include "Normal.h"

using namespace std;

//
// Constructors
//
TourneyPool::TourneyPool() {
  numt = 0;
  numr = 0;
  stats_calculated = false;
  teamnames.clear();
}

TourneyPool::TourneyPool(const TourneyPool& pool)
  : numt(pool.numt), numr(pool.numr),
    A(pool.A), P(pool.P),
    teamnames(pool.teamnames),
    stats_calculated(pool.stats_calculated),
    muX(pool.muX),
    sig2X(pool.sig2X),
    vXX(pool.vXX)
{
}

// construct from ProbSuites
TourneyPool::TourneyPool(const ProbSuite& a, const ProbSuite& p)
  : numt(a.teams()),
    numr(a.rounds()),
    A(a), P(p),
    teamnames()
{
  assert(a.teams() == p.teams());
  stats_calculated = false;
}

//
// Assignment
//
TourneyPool& TourneyPool::operator=(const TourneyPool&) {
  cerr << "TourneyPool::operator= unimplemented" << endl;
  assert(false);
}

//
// Destructor
//
TourneyPool::~TourneyPool() {
}

// set team names
void TourneyPool::set_names(const Names& n) {
  assert(numt == n.size());
  teamnames = n;
}

// expected
//    Calculate the expected return on picks b in this pool, given
//    number of opponents.
double TourneyPool::smooth_expected(int N, const Picks& b,
				    Normal_integrator *nit)
{
  if (!stats_calculated) {
    muX = mean(A,P);
    sig2X = variance(A,P);
    vXX = covariance(A,P,P);
    stats_calculated = true;
  }

  if (nit == NULL) {
    Normal_integrator mynit;
    return mynit.expect_integrate(N,muX,sig2X,
				  mean(A,b), variance(A,b),
				  vXX, covariance(A,P,b));
  } else {
    return nit->expect_integrate(N,muX,sig2X,
				 mean(A,b), variance(A,b),
				 vXX, covariance(A,P,b));
  }
}

/*********************************************************************/
/*  Output  stuff                                                    */
/*********************************************************************/

// display_summary
//    summarize pool data to cout
//
void TourneyPool::display_summary() const {
  cout << "Pool " << mytitle << " with " << numt << " teams" << endl;
  cout << "rounds are worth ";
  for (round_t r = 1; r <= rounds(); r++) cout << worth(r) << ' ';
  cout << endl;
  cout << "a: " << A.title() << endl;
  cout << "p: " << P.title() << endl;
}

// display_picks
//    print a Picks to cout using teamname data
void TourneyPool::display_picks(const Picks& X) const {
  X.display(teamnames);
}

// display_opponent_stats
//    cout stats about opponent scores
void TourneyPool::display_opponent_stats() {
  cout << "opponent score mean: " <<
    mean(A,P) << endl;
  cout << "opponent score standard deviation: " <<
    sqrt(variance(A,P)) << endl;
  cout << "correlation between opponent scores: " <<
    covariance(A,P,P)/
    variance(A,P) << endl;
}

// display_stats
//    cout stats about a set of picks
void TourneyPool::display_stats(Picks X) {
  cout << "actual probability: " << probability(A,X) << endl;
  cout << "actual mean score: " << mean(A,X) << endl;
  cout << "actual score standard deviation: " <<
    sqrt(variance(A,X)) << endl;
  cout << "perceived probability: " << probability(P,X) << endl;
  cout << "perceived mean score: " << mean(P,X) << endl;
  cout << "perceived score standard deviation: " <<
    sqrt(variance(P,X)) << endl;
  cout << "correlation with opponents: " <<
    covariance(A,P,X)/
    sqrt(variance(A,P)*variance(A,X)) << endl;
}

