/*
 * TourneyWinround.C
 *
 *   Manages one set of winround data for a tournament.
 *   Winround data consists of two data sets:
 *       P(i->k) : probability that team i reaches and wins round k
 *       P(i->k && j->l) : probability that team i reaches and wins round k
 *                         and team j reaches and wins round l
 *
 * Copyright (C) 2005 Bryan Clair
 *
 * This file is part of CLOP.
 *
 * CLOP is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * CLOP is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with CLOP; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *
 */

#include <stdio.h>
#include <cassert>

using namespace std;

#include "TourneyBasics.h"
#include "TourneyH2H.h"
#include "TourneyWinround.h"

// Macros to access multi-dimensional arrays of data (quickly)

#define SOLO_ENTRY(t,r) solo[(((r) << numr) + (t))]
#define PAIR_ENTRY(t1,r1,t2,r2)\
      pair[(((((r1) << numr) + (t1)) * (numr+1)) << numr) +\
           (((r2) << numr) + (t2))]

Winround::Winround()
{
  numt = 0;
  numr = 0;
  solo = NULL;
  pair = NULL;
}

Winround::~Winround()
{
  if (solo != NULL) delete solo;
  if (pair != NULL) delete pair;
}

Winround::Winround(const Winround& w) {
  numt = w.numt;
  numr = w.numr;

  solo = new double[numt * (numr+1)];
  for (int i = 0; i < numt * (numr+1); i++)
    solo[i] = w.solo[i];

  pair = new double[numt * (numr+1) * numt * (numr+1)];
  for (int i = 0; i < numt * (numr+1) * numt * (numr+1); i++)
    pair[i] = w.pair[i];
}

Winround& Winround::operator=(const Winround& w) {
  if (this != &w) {
    numt = w.numt;
    numr = w.numr;

    if (solo != NULL) delete solo;
    solo = new double[numt * (numr+1)];
    for (int i = 0; i < numt * (numr+1); i++)
      solo[i] = w.solo[i];
    
    if (pair != NULL) delete pair;
    pair = new double[numt * (numr+1) * numt * (numr+1)];
    for (int i = 0; i < numt * (numr+1) * numt * (numr+1); i++)
      pair[i] = w.pair[i];
  }
  return *this;
}

// constructor
//        From array data
//
//        If wpair is NULL, pair data will be constructed as well
//        as possible from the solo data.  Some pairs are
//        fine (independent or impossible).  Others will be computed
//        using an approximate h2h derived from the wsolo data.
//
Winround::Winround(team_t teams, double *wsolo, double *wpair) {
  // The tricky part here is that the parameter arrays are multidimensional
  // in a different way than the internal solo and pair arrays.
  numt = teams;
  numr = rounds_from_teams(teams);

  solo = new double[numt * (numr+1)];
  pair = new double[numt * (numr+1) * numt * (numr+1)];

  for (team_t i = 0; i < numt; i++) {
    for (round_t r = 0; r <= numr; r++) {
      SOLO_ENTRY(i,r) = wsolo[(numr+1)*i + r];
    }
  }

  if (wpair == NULL) {
    // compute approximate h2h data and go from there
    double approxh2hdata[numt*numt];
    round_t r;
    for (team_t i = 0; i < numt; i++) {
      for (team_t j=0; j < numt; j++) {
	if (i == j) approxh2hdata[i*numt+i] = .5;
	else {
	  r = meet_round(i,j);
	  if (SOLO_ENTRY(i,r-1)*SOLO_ENTRY(j,r-1) == 0) {
	    approxh2hdata[i*numt+j] = .5;  // arbitrary - teems cannot meet
	  } else {
	    // compute the average of team i's chance of advancing
	    // and team j's chance of not advancing
	    approxh2hdata[i*numt+j] = .5 + (SOLO_ENTRY(i,r)*SOLO_ENTRY(j,r-1) -
					SOLO_ENTRY(j,r)*SOLO_ENTRY(i,r-1))/
	      (2*SOLO_ENTRY(i,r-1)*SOLO_ENTRY(j,r-1));
	  }
	}
      }
    }
    HeadToHead approxh2h(numt,approxh2hdata);
    build_pair_from_solo_and_h2h(approxh2h);

  } else {

    // copy given pair data into member array
    for (team_t i = 0; i < numt; i++) {
      for (team_t j=0; j < numt; j++) {
	for (round_t r = 0; r <= numr; r++) {
	  for (round_t s=0; s <= numr; s++) {
	    PAIR_ENTRY(i,r,j,s) =
	      wpair[s + (numr+1)*(r + (numr+1)*(j + numt*i))];
	  }
	}
      }
    }
  }
}

// constructor
//        From head-to-head data
Winround::Winround(const HeadToHead& h2h) {
  numt = h2h.teams();
  numr = rounds_from_teams(numt);

  solo = new double[numt * (numr+1)];
  pair = new double[numt * (numr+1) * numt * (numr+1)];

  // calculate P(i->r) table:
  // P(i->0) = 1
  // P(i->r+1) = P(i->r) * SUM{j opposite i in round r: P(j->r) * h2h(i,j)}

  for (team_t i=0; i<numt; i++)
    SOLO_ENTRY(i,0) = 1;
  for (round_t r = 1; r <= numr; r++) {
    for (team_t i = 0; i < numt; i++) {
      double sum = 0;
      // j runs through the sub-bracket opposite team i for round r.
      FOR_EACH_OPPONENT(j,i,r) {
	sum += SOLO_ENTRY(j,r-1) * h2h(i,j);
      }
      SOLO_ENTRY(i,r) = SOLO_ENTRY(i,r-1) * sum;
    }
  }

  build_pair_from_solo_and_h2h(h2h);
}

// build_pair_from_solo_and_h2h
//    Inductive computation of P(i->r && j->s), given P(i->r) data and h2h data
//
void Winround::build_pair_from_solo_and_h2h(const HeadToHead& h2h) {
  numt = h2h.teams();
  numr = rounds_from_teams(numt);

  // Begin inductive computation of PAIR_ENTRY table
  // P(i->0 && j->0) == 1
  for (team_t i = 0; i < numt; i++) {
    for (team_t j=0; j < numt; j++) {
      PAIR_ENTRY(i,0,j,0) = 1;
    }
  }

  // Inductive steps.. compute all terms with r >= s
  for (round_t r = 1; r <= numr; r++) {
    for (team_t i = 0; i < numt; i++) {
      for (round_t s=0; s <= r; s++) {
	for (team_t j=0; j < numt; j++) {
	  round_t m = meet_round(i,j);
	  if (i == j) {
	    // same team
	    PAIR_ENTRY(i,r,j,s) = SOLO_ENTRY(i,r);
	  } else if (s >= m) {
	    // i->r && j->s is impossible
	    PAIR_ENTRY(i,r,j,s) = 0;
	  } else if (r < m) {
	    // i->r && j->s are independent
	    PAIR_ENTRY(i,r,j,s) = SOLO_ENTRY(i,r) * SOLO_ENTRY(j,s);
	  } else if (SOLO_ENTRY(i,r-1) == 0) {
	    // i->r && j->s is impossible -- had to check so not do div by 0
	    PAIR_ENTRY(i,r,j,s) = 0;
	  } else if (r > m) {
	    // interaction between i and j was already dealt with, so
	    // induct back.. note that r > m > s, so r-1 >= s here.
	    PAIR_ENTRY(i,r,j,s) = PAIR_ENTRY(i,r-1,j,s)
	      * SOLO_ENTRY(i,r) / SOLO_ENTRY(i,r-1);
	  } else { // (r == m) && (s < m)
	    // induct back.. again, r = m > s so r-1 >= s here.
	    double sum = 0;
	    // k runs through the sub-bracket opposite team i for round r.
	    FOR_EACH_OPPONENT(k,i,r) {
	      sum += h2h(i,k) * PAIR_ENTRY(k,r-1,j,s);
	    }
	    PAIR_ENTRY(i,r,j,s) = SOLO_ENTRY(i,r-1) * sum;
	  }
	}
      }
    }
  }

  // Finish PAIR_ENTRY table.. terms with r < s:
  for (round_t s=1; s <= numr; s++) {
    for (team_t i = 0; i < numt; i++) {
      for (round_t r = 0; r < s; r++) {
	for (team_t j=0; j < numt; j++) {
	  PAIR_ENTRY(i,r,j,s) = PAIR_ENTRY(j,s,i,r);
	}
      }
    }
  }
}

// operator()
//        Return probability that team t reaches and wins in round r.
//        0 <= t < teams()
//        0 <= r <= rounds()  (always 1 for r==0)
inline double Winround::operator() (team_t t, round_t r) const {
  assert(0 <= t && t < teams());
  assert(0 <= r && r <= rounds());
  return SOLO_ENTRY(t,r);
}

// operator()
//        Return probability that team t1 reaches and wins in round r1
//        and team t2 reaches and wins in round r2.
//        0 <= ti < teams()
//        0 <= ri <= rounds()
inline double Winround::operator()
  (team_t t1, round_t r1, team_t t2, round_t r2)
  const
{
  assert(0 <= t1 && t1 < teams());
  assert(0 <= r1 && r1 <= rounds());
  assert(0 <= t2 && t2 < teams());
  assert(0 <= r2 && r2 <= rounds());
  return PAIR_ENTRY(t1,r1,t2,r2);
}

// dump_solo
//        Dump only the P(i->r) data to stdout as a stream of floats.
void Winround::dump_solo() const {
  for (team_t i=0; i < teams(); i++) {
    printf("  ");
    for (round_t k = 0; k <= rounds(); k++) {
	printf("%.3f ",(*this)(i,k));
    }
    printf("\n");
  }
}

// dump_pair
//        Dump only the the P(i->r && j->s) data to stdout as a
//        stream of floats.
void Winround::dump_pair() const {
  for (team_t i = 0; i < teams(); i++) {
    for (team_t j = 0; j < teams(); j++) {
      printf("\n");
      for (round_t r = 0; r <= rounds(); r++) {
	printf("  ");
	for (round_t s = 0; s <= rounds(); s++) {
	  printf("%.3f ",(*this)(i,r,j,s));
	}
	printf("\n");
      }
    }
  }
}


// display
//        Dump only the P(i->r) data to stdout, in a human friendly format.
//        Use teamnames if given.
void Winround::display(char *teamnames[]) {
  team_t i;
  round_t k;
  
  printf("Team ");
  for (k = 0; k <= rounds(); k++) {
    printf(" R%2d  ",k);
  }
  printf("\n");
  
  for (i=0; i < teams(); i++) {
    if (teamnames != NULL)
      printf("[%s]",teamnames[i]);
    else
      printf("[%2d] ",i);
    for (k = 0; k <= rounds(); k++) {
	printf("%.3f ",(*this)(i,k));
    }
    printf("\n");
  }
}

// display_all
//        Dump the P(i->r) and the P(i->r && j->s) data to stdout.
//        Use teamnames if given.
void Winround::display_all(char *teamnames[]) {
  display(teamnames);

  printf("Probablities of both teams advancing to given rounds\n");
  for (team_t i = 0; i < teams(); i++) {
    for (team_t j = 0; j < teams(); j++) {
      if (teamnames != NULL)
	printf("---  [%s] && [%s]  ---\n",teamnames[i],teamnames[j]);
      else
	printf("---  [%2d] && [%2d]  ---\n",i,j);
      for (round_t r = 0; r <= rounds(); r++) {
	for (round_t s = 0; s <= rounds(); s++) {
	  printf("%.3f ",(*this)(i,r,j,s));
	}
	printf("\n");
      }
    }
  }
}
