/*
 * TourneyWinround.h
 *
 *   Manages one set of winround data for a tournament.
 *   Winround data consists of two data sets:
 *       P(i->k) : probability that team i reaches and wins round k
 *       P(i->k && j->l) : probability that team i reaches and wins round k
 *                         and team j reaches and wins round l
 *
 * Copyright (C) 2005 Bryan Clair
 *
 * This file is part of CLOP.
 *
 * CLOP is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * CLOP is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with CLOP; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *
 */

#ifndef _TOURNEYWINROUND_H_
#define _TOURNEYWINROUND_H_

#include "TourneyBasics.h"
#include "TourneyH2H.h"

class Winround {
 public:
  Winround();                           // default constructor
  ~Winround();                          // destructor
  Winround(const Winround&);            // copy constructor
  Winround& operator=(const Winround&); // assignment

  // constructor
  //        From array data
  //
  //        The wsolo array is size (T * (R+1)), in order:
  //        P(0->0) P(0->1) ... P(0->R) P(1->0) ... P((T-1)->R)
  //
  //        The wpair array is, (T * T * (R+1) * (R+1)), in order:
  //        P(0->0 & 0->0) P(0->0 & 0->1) .. P(0->0 & 0->R)
  //        P(0->1 & 0->0) ..                P(0->1 & 0->R)
  //         ...
  //        P(0->R & 0->0) ..                P(0->R & 0->R)
  //        P(0->0 & 1->0) P(0->0 & 1->1) .. P(0->0 & 1->R)
  //         ...
  //        P(0->R & 1->0) ..                P(0->R & 1->R)
  //         ...
  //        P(0->R & (T-1)->0) ..        P(0->R & (T-1)->R)
  //        P(1->0 & 0->0) ...
  //                                 P((T-1)->R & (T-1)->R)
  //
  //        If wpair is unspecified, it will be constructed as well
  //        as possible from the incomplete data.  Some pairs are
  //        fine (independent or impossible).  Others will be computed
  //        using an approximate h2h derived from the wsolo data.
  //
  Winround(team_t teams, double *wsolo, double *wpair = NULL);

  // constructor
  //        From head-to-head data
  Winround(const HeadToHead&);

  // rounds
  // teams
  //        Return number of rounds/teams
  inline round_t rounds(void) const { return numr; }
  inline team_t teams(void) const { return numt; }

  // operator() (t,r)
  //        Return probability that team t reaches and wins in round r.
  //        0 <= t < teams()
  //        0 <= r <= rounds()  (always 1 for r==0)
  double operator() (team_t t, round_t r) const;

  // operator() (t1,r1,t2,r2)
  //        Return probability that team t1 reaches and wins in round r1
  //        and team t2 reaches and wins in round r2.
  //        0 <= ti < teams()
  //        0 <= ri <= rounds()
  double operator() (team_t t1, round_t r1, team_t t2, round_t r2) const;

  // dump_solo
  //        Dump only the P(i->r) data to stdout as a stream of floats.
  void dump_solo() const;

  // dump_pair
  //        Dump only the the P(i->r && j->s) data to stdout as a
  //        stream of floats.
  void dump_pair() const;

  // display
  //        Dump only the P(i->r) data to stdout, in a human friendly format.
  //        Use teamnames if given.
  void display(char *teamnames[] = NULL);

  // display_all
  //        Dump the P(i->r) and the P(i->r && j->s) data to stdout.
  //        Use teamnames if given.
  void display_all(char *teamnames[] = NULL);

 private:
  team_t numt;
  round_t numr;
  double *solo;
  double *pair;

  // build_pair_from_solo_and_h2h
  //    Inductive computation of P(i->r && j->s), given P(i->r) data and h2h data
  void build_pair_from_solo_and_h2h(const HeadToHead&);
};

#endif
