/*
 * fall.C
 *
 *    Find and output the expected return for all bets in a football pool.
 *
 * usage: fall [-t<threads>] [-q] [-n<competitors>] datafile [-Aactuals] [-Pperceiveds]
 *    -q   Quiet
 *    -n   Number of competitors
 *    -t   Number of threads
 *    -AP  change column of data file
 *
 * Copyright (C) 2005 Bryan Clair
 *
 * This file is part of CLOP.
 *
 * CLOP is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * CLOP is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with CLOP; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *
 *   9/29/16 Fixing obsolete code so it will compile.
 */

#include <iostream>
#include <fstream>
#include <stdlib.h>
#include <time.h>
#include <pthread.h>
#include <unistd.h>

#include "FootballPack.h"
#include "FootballPool.h"

using namespace std;

//
// usage
//
void usage(const char *msg=NULL) {
  cerr << "usage: fall [-t<threads>] [-q] [-n<competitors>] datafile [-Aactuals] [-Pperceiveds]" << endl;
  if (msg) cerr << "       " << msg << endl;
  exit(1);
}

//
// default arguments
//
long competitors = 1000;
bool quiet = false;
int NUM_THREADS=2;


//
// struct work_data
//    shared struct for worker threads
//
struct work_data {
  pthread_mutex_t lock;
  int N;  // # of competitors
  FootballPool *P;  // pool data
  bool done;
  Outcome bet;   // the next bet to work on
};

pthread_mutex_t stdout_lock;
  
//
// worker
//    worker thread calculates expected value of bets and
//    prints results to cout.
//
void *worker(void *v) {
  struct work_data *wd = (struct work_data *)v;

  Outcome mybet;
  double myval;

  pthread_mutex_lock(&wd->lock);

  while (!wd->done) {
    mybet = wd->bet--;
    if (mybet == 0) wd->done = true;

    pthread_mutex_unlock(&wd->lock);

    myval = (wd->P)->expected(wd->N,mybet);

    pthread_mutex_lock(&stdout_lock);
    cout << myval << '\t';
    wd->P->displayOutcome(mybet);
    cout << endl;
    pthread_mutex_unlock(&stdout_lock);

    pthread_mutex_lock(&wd->lock);
  }

  pthread_mutex_unlock(&wd->lock);
}

//
// main
//
main(int argc, char *argv[]) {
  // Parse arguments
  std::string atype = "", ptype = "";

  char opt;
  while ((opt = getopt(argc,argv,"qn:t:A:P:")) != -1)
    switch (opt) {
    case 'q':
      quiet = true;
      break;
    case 't':
      NUM_THREADS = atoi(optarg);
      if (NUM_THREADS < 1) usage("bad thread count");
      break;
    case 'n':
      competitors = atol(optarg);
      break;
    case 'A':
      atype = optarg;
      break;
    case 'P':
      ptype = optarg;
      break;
    default:
      usage();
    }
  if (competitors <= 0) usage("# of competitors must be > 0");
  if (optind != argc-1) usage();

  // Open file and read pool data
  char *datafilename = argv[optind];
  ifstream datafile(datafilename);
  if (!datafile) {
    perror(datafilename);
    exit(1);
  }

  FootballPool pool;
  try {
    pool.read(datafile,atype,ptype);
  } catch (FP_read_error& err) {
    err.display();
    exit(1);
  }

  // Print header
  if (!quiet) {
    pool.write(cerr);
    cerr << "File '" << datafilename << "' with "
	 << competitors << " competitors.\n";
  }

  // Calculate for all bets

  // Initialize shared data
  struct work_data wd;
  pthread_mutex_init(&stdout_lock,NULL);
  pthread_mutex_init(&wd.lock,NULL);
  wd.N = competitors;
  wd.P = &pool;
  wd.done = false;
  Outcome maxbet = (1 << pool.games()) - 1;
  wd.bet = maxbet;

  // Create worker threads
  pthread_t workers[NUM_THREADS];
  int i;
  for (i=0; i<NUM_THREADS; i++) {
    pthread_create(&workers[i],NULL,worker,&wd);
  }

  // Periodically report on progress
  if (!quiet) {
    time_t start = time(NULL);
    while (!wd.done) {
      sleep(20);
      cerr << "Total time: " << (time(NULL) - start)/60 << " minutes.  ";
      cerr << (100*(maxbet - wd.bet))/(maxbet + 1) << "% complete.  ";
      cerr << (wd.bet * (time(NULL) - start))/(60*(maxbet - wd.bet))
	   << " minutes left.\n";
    }
  }

  // Wait for workers to finish (not long if wd.bet == 0)
  for (i=0; i<NUM_THREADS; i++) {
    pthread_join(workers[i],NULL);
  }
}

