/*
 * fseek.C
 *
 *   Greedy seek for good bets in a football pool.
 *
 * usage: fseek [-q] [-t<threads>] [-n<competitors>] datafile [-Aactuals] [-Pperceiveds]
 *    -q   Quiet
 *    -n   Number of competitors
 *    -t   Number of threads when seeking
 *    -AP  change column of data file
 *    
 * Copyright (C) 2005 Bryan Clair
 *
 * This file is part of CLOP.
 *
 * CLOP is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * CLOP is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with CLOP; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *
 *
 * Revisions:
 *   1/9/06 Bryan Clair:
 *       Now starts with "quick" picks instead of random.
 *       Fixed a bug with the 2-away neighbor search, and sped it up.
 *       Removed -S and -z options.
 *   9/29/16 Fixing obsolete code so it will compile.
 */

#include <iostream>
#include <fstream>
#include <pthread.h>
#include <unistd.h>

#include "FootballPack.h"
#include "FootballPool.h"

using namespace std;


//
// usage
//
void usage(const char *msg=NULL) {
  cerr << "usage: fseek [-q] [-t<threads>] [-n<competitors>] datafile [-Aactuals] [-Pperceiveds]" << endl;
  if (msg) cerr << "       " << msg << endl;
  exit(1);
}

//
// default arguments
//
long competitors = 1000;
bool quiet = false;
int NUM_THREADS=2;

//
// struct nbr_search_data
//    shared struct for worker threads
//
struct nbr_search_data {
  pthread_mutex_t lock;
  int N;  // # of competitors
  FootballPool *P;  // pool data
  Outcome start;   // the center point to find neighbors of
  Outcome dir;  // next direction to move - stop when zero
  Outcome top_o;  // best neighbor so far
  double top_v;  // best neighbor's value
};
  
//
// best_nbr_worker
//    worker thread checks neighbors of an Outcome while
//    there are still neighbors to check.
//
void *best_nbr_worker(void *v) {
  struct nbr_search_data *sd = (struct nbr_search_data *)v;

  Outcome nbr;
  double nbr_v;

  pthread_mutex_lock(&sd->lock);

  while (sd->dir > 0) {
    nbr = sd->start ^ sd->dir;
    sd->dir = (sd->dir >> 1);

    pthread_mutex_unlock(&sd->lock);

    nbr_v = (sd->P)->expected(sd->N,nbr);

    pthread_mutex_lock(&sd->lock);

    if (nbr_v > sd->top_v) {
      sd->top_v = nbr_v;
      sd->top_o = nbr;
    }
  }

  pthread_mutex_unlock(&sd->lock);
}

//
// best_nbr
//   Find the best bet in pool P with N competitors which is exactly
//   one game different from start.
//
//   The start_dir parameter can be used to limit the number of directions
//   searched.  It defaults to search all directions.
//
//   Returns the best bet found, and the expected return for that bet in bval.
//
Outcome best_nbr(int N, FootballPool &P, Outcome start, double *bval,
		 Outcome start_dir = 0)
{
  struct nbr_search_data sd;
  pthread_t workers[NUM_THREADS];

  pthread_mutex_init(&sd.lock,NULL);
  sd.N = N;
  sd.P = &P;
  sd.start = start;
  sd.dir = (start_dir > 0) ? start_dir : (1 << (P.games()-1));
  sd.top_v = 0;

  int i;
  for (i=0; i<NUM_THREADS; i++) {
    pthread_create(&workers[i],NULL,best_nbr_worker,&sd);
  }

  for (i=0; i<NUM_THREADS; i++) {
    pthread_join(workers[i],NULL);
  }

  *bval = sd.top_v;
  return sd.top_o;
}

//
// seek_best
//   Seek towards a good bet in pool P with N competitors, beginning the
//   search at location start.
//   Returns the bet where the search stopped, and the expected return for
//   that bet in bval.
//
//   Each step moves to a the best bet with one game changed from the current.
//   If doubleseeking is true, tries two-away after one-away max is attained.
//
Outcome seek_best(int N, FootballPool& P, Outcome start, double *bval) {
  Outcome top_o, nbr, nbr2, dir;
  double top_v, nbr_v, nbr2_v;
  bool moved;

  top_o = start;
  top_v = P.expected(N, top_o);
    
  do {
    if (!quiet) {
      cout << top_v << "\t";
      P.displayOutcome(top_o);
      cout << endl;
    }

    moved = false;

    // try all one-away neighbors

    nbr = best_nbr(N, P, top_o, &nbr_v);
    if (nbr_v > top_v) {
	  top_o = nbr;
	  top_v = nbr_v;
	  moved = true;
    }
    
    // try all two-away neighbors, if needed

    if (!moved) {
      nbr = top_o;
      for (dir = (1 << (P.games()-1)); dir > 1; dir = (dir >> 1)) { 
	if (!quiet) { cout << '.'; cout.flush(); }

	nbr ^= dir; // flip bit to move to a neighbor
	nbr2 = best_nbr(N, P, nbr, &nbr2_v, dir >> 1);
	if (nbr2_v > top_v) {
	  top_o = nbr2;
	  top_v = nbr2_v;
	  moved = true;
	}
	nbr ^= dir; // flip bit back before doing next neighbor
      }
      if (!quiet) cout << endl;
    }
  } while (moved);

  *bval = top_v;
  return top_o;
}

//
// main
//
main(int argc, char *argv[]) {
  // Parse arguments
  string atype = "", ptype = "";
  char opt;
  while ((opt = getopt(argc,argv,"qn:t:A:P:")) != -1)
    switch (opt) {
    case 'A':
      atype = optarg;
      break;
    case 'P':
      ptype = optarg;
      break;
    case 'q':
      quiet = true;
      break;
    case 'n':
      competitors = atol(optarg);
      break;
    case 't':
      NUM_THREADS = atoi(optarg);
      if (NUM_THREADS < 1) usage("bad thread count");
      break;
    default:
      usage();
    }
  if (competitors <= 0) usage("# of competitors must be > 0");
  if (optind != argc-1) usage();

  // Open file and read pool data
  char *datafilename = argv[optind];
  ifstream datafile(datafilename);
  if (!datafile) {
    perror(datafilename);
    exit(1);
  }
  if (!quiet) {
    cout << "Pool in file " << datafilename << " with "
	 << competitors << " competitors.\n";
  }
  FootballPool pool;
  try {
    pool.read(datafile,atype,ptype);
  } catch (FP_read_error& err) {
    err.display();
    exit(1);
  }

  // Data successfully read
  if (!quiet) pool.write(cout);

  double bval;
  Outcome best = seek_best(competitors, pool,
		   quick_picks(competitors,pool.actual(), pool.perceived()),
		   &bval);
			
  // Report results
  if (!quiet) cout << "Best:" << endl;

  cout << bval << '\t';
  pool.displayOutcome(best);
  cout << endl;
}

