/*
 * fstats.C
 *
 *    Calculate stats (expected return, probability) for picks read from stdin.
 *
 * usage: fstats [-qnAPsdgv] datafile
 *   -q  quiet.. no headers
 *   -n<competitors>
 *   -AP change column of data file
 *   -s  use smooth approximation to calculate expected
 *   -d  show detailed statistics
 *   -g  show vertical game-by-game data
 *   -v<actual spread>  show variation of expected wrt actuals
 *   -w<perceived spread> show variation of expected wrt perceiveds
 *
 * Copyright (C) 2005 Bryan Clair
 *
 * This file is part of CLOP.
 *
 * CLOP is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * CLOP is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with CLOP; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *
 * Revisions:
 *   1/4/06  Corrected and improved usage statement.
 *   1/16/06 Added display of picks by column to detailed option 
 *   6/29/06 Added -w option for perceiveds to go with -v for actuals
 *   9/29/16 Fixing obsolete code so it will compile.
 */

#include <iostream>
#include <iomanip>
#include <fstream>
#include <sstream>
#include <stdlib.h>
#include <time.h>
#include <unistd.h>
#include <gsl/gsl_statistics.h>

#include "PoolMath.h"
#include "FootballPack.h"
#include "FootballPool.h"

using namespace std;

//
// global pool
//
FootballPool pool;

//
// usage
//
void usage(const char *msg=NULL) {
  cerr << "usage: fstats [-qnAPsdgv] datafile" << endl;
  cerr << "      -q  quiet.. no headers" << endl;
  cerr << "      -n<competitors>" << endl;
  cerr << "      -AP change column of data file" << endl;
  cerr << "      -s  use smooth approximation to calculate expected" << endl;
  cerr << "      -d  show detailed statistics" << endl;
  cerr << "      -g  show vertical game-by-game data" << endl;
  cerr << "      -v<actual spread>  show variation of expected wrt actuals" << endl;
  cerr << "      -w<perceived spread> show variation of expected wrt perceiveds" << endl;
  if (msg) cerr << "    " << msg << endl;
  exit(1);
}

//
// default arguments
//
long competitors = 1000;
bool quiet = false;
bool smooth = false;
bool detailed = false;
bool gamebygame = false;
bool vary_a = false;
bool vary_p = false;
double actual_spread;  // % variation in actuals for spread calc
double perceived_spread;  // % variation in perceiveds for spread calc
int spread_iterations=200; // # of samples when calculating spread

//
// expected_spread
//   Calculate the spread of the expected return over
//   a range of actuals
//
void expected_spread(Outcome b)
{
  int i,t;
  double evals[spread_iterations];

  srandom(time(NULL));

  cout << "spreads";  cout.flush();
  ProbVec av = pool.actual();
  ProbVec pv = pool.perceived();

  for (t=0; t < spread_iterations; t++) {
    if ((!quiet)&&(t % 10 == 0))
      { cout << "."; cout.flush(); }
    for (i=0; i<pool.games(); i++) {
      if (vary_a)
	av[i] = distort_prob_uniform(pool.actual()[i],actual_spread);
      if (vary_p)
	pv[i] = distort_prob_uniform(pool.perceived()[i],perceived_spread);
    }
    evals[t] = expected(competitors,av,pv,b);
  }
  cout << endl;
  cout << "expected at ";
  if (vary_a) cout << "(" << actual_spread << " variation of actuals) " << endl;
  if (vary_p) cout << "(" << perceived_spread << " variation of perceiveds) " << endl;
  cout << "\tmean\t" << gsl_stats_mean(evals,1,spread_iterations) << endl;
  cout << "\ts.d.\t" << sqrt(gsl_stats_variance(evals,1,spread_iterations)) << endl;
  cout << "\tsd/mean\t" << sqrt(gsl_stats_variance(evals,1,spread_iterations))/
    gsl_stats_mean(evals,1,spread_iterations) << endl;
  cout << "\tskew\t" << gsl_stats_skew(evals,1,spread_iterations) << endl;
  cout << "\tkurt.\t" << gsl_stats_kurtosis(evals,1,spread_iterations) << endl;
}

// display_stats
//   Calculated statistics for one vector of probabilities and an outcome.
void display_stats(const ProbVec& x, Outcome b)
{
  cout << "\tprob: " << probability(x,b);
  cout << "\tmean: " << mean(x,b);
  cout << "\tvar: " << variance(x,b) << endl;
  cout << "\tupsets: " <<
      x.games() - scoreOutcome(b,x.favorites(),x.games()) << endl;
}

//
// one set of picks
//
void do_one_set(char *line) {
  Outcome X;
  istringstream pix(line);
  int g;
  
  try {
    X = pool.readOutcome(pix);
  } catch (FP_read_error& err) {
    err.display();
    return;
  }
  
  g = pool.games();
  
  if (!(gamebygame && quiet && !detailed)) {
    if (!detailed) {
      if (smooth)
	cout << pool.smooth_expected(competitors,X) << '\t';
      else
	cout << pool.expected(competitors,X) << '\t';
    }
    
    pool.displayOutcome(X);
    cout << endl;
  }

  if (detailed) {
    cout << "Teams picked:\t";
    printOutcome(X,g);
    cout << '\n';
    cout << "\texp:  " << pool.expected(competitors,X) << '\n';
    cout << "\tsexp: " << pool.smooth_expected(competitors,X) << '\n';
    
    cout << pool.actual_title() << endl;
    display_stats(pool.actual(),X);

    cout << pool.perceived_title() << endl;
    display_stats(pool.perceived(),X);
  }

  if (vary_a || vary_p) {
    expected_spread(X);
  }

  if (gamebygame) {
    // ouput results as a comma separated list of triples {a,p,w}
    // where a,p are actual & perceived for the actual favorite (a >= .5)
    // and w=1 for picking favorite, 0 for picking underdog.
    if (!quiet) cout << "Pick\tA\tP\tdE/dai\tdE/dpi" << endl;
    for (int i = 0; i<pool.games(); i++) {
      if (pool.actual()[i] > .5) {
	cout << gameOutcome(X,i) << '\t'
	     << pool.actual()[i] << '\t' << pool.perceived()[i] << '\t'
	     << dexpected_dxi(competitors,pool.actual(),pool.perceived(),X,true,i) << '\t'
	     << dexpected_dxi(competitors,pool.actual(),pool.perceived(),X,false,i) << '\t'
	     << endl;
      } else {
	cout << 1-gameOutcome(X,i) << '\t'
	     << 1-pool.actual()[i] << '\t' << 1-pool.perceived()[i] << '\t' 
	     << -dexpected_dxi(competitors,pool.actual(),pool.perceived(),X,true,i) << '\t'
	     << -dexpected_dxi(competitors,pool.actual(),pool.perceived(),X,false,i) << '\t'
	     << endl;
      }
    }
  }
}

//
// main
//
main(int argc, char *argv[]) {
  // Parse arguments
  string atype = "", ptype = "";
  
  char opt;
  while ((opt = getopt(argc,argv,"qsdgv:w:n:A:P:")) != -1)
    switch (opt) {
    case 'A':
      atype = optarg;
      break;
    case 'P':
      ptype = optarg;
      break;
    case 'n':
      competitors = atol(optarg);
      break;
    case 'q':
      quiet = true;
      break;
    case 'g':
      gamebygame = true;
      break;
    case 'd':
      detailed = true;
      break;
    case 's':
      smooth = true;
      break;
    case 'v':
      vary_a = true;
      actual_spread = atof(optarg);
      if ((actual_spread <= 0) || (actual_spread >= 1))
	usage("actual_spread must be in [0,1]");
      break;
    case 'w':
      vary_p = true;
      perceived_spread = atof(optarg);
      if ((perceived_spread <= 0) || (perceived_spread >= 1))
	usage("perceived_spread must be in [0,1]");
      break;
    default:
      usage();
    }
  if (competitors <= 0) usage("# of competitors must be > 0");
  if (optind != argc-1) usage();

  // Open file and read pool data
  char *datafilename = argv[optind];
  ifstream datafile(datafilename);
  if (!datafile) {
    perror(datafilename);
    exit(1);
  }

  try {
    pool.read(datafile,atype,ptype);
  } catch (FP_read_error& err) {
    err.display();
    exit(1);
  }

  // Print header
  if (!quiet) {
    cout << datafilename << " (" << pool.games() << " games)"
	 << " with " << competitors << " competitors.\n";
  }

  char line[256];
  while(cin.getline(line,256)) {
    do_one_set(line);
  }
}

