/*
 * pcalc.C
 *
 *    Calcualate a table of winround data from a list of picks.
 *
 *  usage: pcalc [-r<rounds>] [picksfile]
 *
 *    Expects one pick per line on stdin in depth format.
 *    Outputs data suitable for reading into a TourneyPool.
 *
 * Copyright (C) 2005 Bryan Clair
 *
 * This file is part of CLOP.
 *
 * CLOP is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * CLOP is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with CLOP; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *
 *   9/29/16 Fixing obsolete code so it will compile.
 */

#include <iostream>
#include <iomanip>
#include <fstream>
#include <sstream>
#include <cstdlib>
#include <unistd.h>

using namespace std;

#include "TourneyBasics.h"

//
// usage
//
void usage(const char *msg=NULL) {
  cerr << "usage: pcalc [-r<rounds>] [picksfile]" << endl;
  if (msg) cerr << "       " << msg << endl;
  exit(1);
}

//
// default arguments
//
team_t teams;
round_t rounds = 6;

//
// Global counts
//
int *wsolo;
int *wpair;
int wcount;

// Macros to access multi-dimensional arrays of data (quickly)
#define SOLO_ENTRY(t,r) wsolo[(((r) << rounds) + (t))]
#define PAIR_ENTRY(t1,r1,t2,r2)\
      wpair[(((((r1) << rounds) + (t1)) * (rounds+1)) << rounds) +\
           (((r2) << rounds) + (t2))]

//
// one set of picks
//    return true if successfully read & counted
//
bool do_one_set(char *line) {
  istringstream s(line);
  int pix[teams];

  for (team_t i = 0; i < teams; i++) {
    s >> pix[i];
    if ((pix[i] < 1) || (pix[i] > rounds+1)) {
      cerr << "bad round number: " << pix[i] << endl;
      return false;
    }
    pix[i];
  }

  // solo count
  for (team_t i = 0; i < teams; i++) {
    for (round_t r = 0; r < pix[i]; r++) {
      SOLO_ENTRY(i,r)++;
    }
  }

  // pair count
  for (team_t i = 0; i < teams; i++) {
    for (team_t j = 0; j < teams; j++) {
      for (round_t r = 0; r < pix[i]; r++) {
	for (round_t s = 0; s < pix[j]; s++) {
	  PAIR_ENTRY(i,r,j,s)++;
	}
      }
    }
  }

  return true;
}

//
// dump results
//
void dump_results(string description) {
  cout << "winround " << teams << " " << description << endl;

  cout << "solo" << endl;
  for (team_t i = 0; i < teams; i++) {
    for (round_t r = 0; r <= rounds; r++)
      cout << setw(10) << SOLO_ENTRY(i,r)/(double)wcount << " ";
    cout << endl;
  }

  cout << endl;
  cout << "pair" << endl;
  for (team_t i = 0; i < teams; i++) {
    for (team_t j = 0; j < teams; j++) {
      for (round_t r = 0; r <= rounds; r++) {
	for (round_t s = 0; s <= rounds; s++) {
	  cout << setw(10) << PAIR_ENTRY(i,r,j,s)/(double)wcount << " ";
	}
	cout << endl;
      }
      cout << endl;
    }
  }
}

//
// read_picks
//    read and count picks from given file
//
void read_picks(istream &f) {
  char line[256];
  while(f.getline(line,256)) {
    if (do_one_set(line)) wcount++;
  }
  cerr << "Read " << wcount << " picks." << endl;
}

//
// main
//
main(int argc, char *argv[]) {
  // Parse arguments

  char *filename = NULL;

  char opt;
  while ((opt = getopt(argc,argv,"r:")) != -1)
    switch (opt) {
    case 'r':
      rounds = atoi(optarg);
      break;
    default:
      usage();
    }

  teams = teams_from_rounds(rounds);

  wsolo = new int[teams * (rounds + 1)];
  for (team_t i = 0; i < teams; i++)
    for (round_t r = 0; r <= rounds; r++)
      SOLO_ENTRY(i,r) = 0;

  wpair = new int[teams * (rounds + 1) * teams * (rounds + 1)];
  for (team_t i = 0; i < teams; i++)
    for (round_t r = 0; r <= rounds; r++)
      for (team_t j = 0; j < teams; j++)
	for (round_t s = 0; s <= rounds; s++)
	  PAIR_ENTRY(i,r,j,s) = 0;

  wcount = 0;

  if (optind == argc-1) {
    // Open file and read pool data
    filename = argv[optind];
    ifstream picksfile(filename);
    if (!picksfile) {
      perror(filename);
      exit(1);
    }
    read_picks(picksfile);
  } else if (optind == argc) {
    read_picks(cin);
  } else {
    usage();
  }

  dump_results(filename ? filename : "stdin");

  delete[] wsolo;
  delete[] wpair;
}

