/*
 * pix2tex.C
 *
 *    Convert picks to LaTeX output
 *
 * usage: pix2tex [-h<height>] [-w<width>] namesfile
 *
 * Copyright (C) 2005 Bryan Clair
 *
 * This file is part of CLOP.
 *
 * CLOP is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * CLOP is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with CLOP; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *
 * 9/29/16 Fixing obsolete code so it will compile.
 */

#include <iostream>
#include <sstream>
#include <fstream>
#include <stdlib.h>
#include <string>
#include <unistd.h>

#include "TourneyPack.h"
#include "TourneyPool.h"
#include "TourneyFile.h"

using namespace std;

//
// usage
//
void usage(char *msg=NULL) {
  cerr << "usage: pix2tex [-h<height>] [-w<width>] namesfile" << endl;
  if (msg) cerr << "       " << msg << endl;
  exit(1);
}

//
// globals
//
Names names;
string title;

//
// size tweaks
//

float pic_height = 224;
float pic_width = 348;
float lr_gap_factor = .6; // extra space between l&r sides
float name_up_factor = .1;  // raise name over line by this factor

float spot_width;  // length of line for one team
float spot_height; // height of smallest gap for one team

//
// r - log2 of # of teams to draw on one side
// (x0,y0) - origin
// x1 - right edge of window
// vspc - vertical space between lines
//
void bracket(const Picks &p, round_t r, float x0, float x1, float y0, float vspc) {
  if (r == 0) {
    // center
    printf("    \\put(%.2f,%.2f){\\line(1,0){%.2f}}\n",
           x0 + spot_width * lr_gap_factor/2,
           y0,
           spot_width
           );
    printf("    \\put(%.2f,%.2f){\\makebox(0,0)[b]{%s}}\n",
           x0 + spot_width * (.5 + lr_gap_factor/2),
           y0 + spot_height*name_up_factor,
           names[p.winner(p.rounds(),0)].c_str());
    // left
    printf("    \\put(%.2f,%.2f){\\line(1,0){%.2f}}\n",
           x0, y0 + vspc*3/16,
           spot_width
           );
    printf("    \\put(%.2f,%.2f){%s}\n",
           x0, y0 + vspc*3/16 + spot_height*name_up_factor,
           names[p.winner(p.rounds()-1,0)].c_str());
    // right
    printf("    \\put(%.2f,%.2f){\\line(-1,0){%.2f}}\n",
           x1, y0 - vspc*3/16,
           spot_width
           );
    printf("    \\put(%.2f,%.2f){\\makebox(0,0)[br]{%s}}\n",
           x1, y0 - vspc*3/16 + spot_height*name_up_factor,
           names[p.winner(p.rounds()-1,1)].c_str());
  } else {
    // left half lines
    printf("    \\multiput(%.2f,%.2f)(0,%.2f){%d}{\\line(1,0){%.2f}}\n",
           x0,y0,
           vspc,
           1 << r,
           spot_width);
    printf("    \\multiput(%.2f,%.2f)(0,%.2f){%d}{\\line(0,1){%.2f}}\n",
           x0+spot_width,y0,
           2*vspc,
           1 << (r-1),
           vspc);
    // right half lines
    printf("    \\multiput(%.2f,%.2f)(0,%.2f){%d}{\\line(-1,0){%.2f}}\n",
           x1,y0,
           vspc,
           1 << r,
           spot_width);
    printf("    \\multiput(%.2f,%.2f)(0,%.2f){%d}{\\line(0,1){%.2f}}\n",
           x1-spot_width,y0,
           2*vspc,
           1 << (r-1),
           vspc);
    
    // left half teams
    for (team_t i = 0; i < 1 << r; i++) {
      printf("    \\put(%.2f,%.2f){%s}\n",
             x0,y0 + i*vspc + spot_height*name_up_factor,
             names[p.winner(p.rounds()-r-1,(1 << r) - i - 1)].c_str());
    }
    // right half teams
    for (team_t i = 0; i < 1 << r; i++) {
      printf("    \\put(%.2f,%.2f){\\makebox(0,0)[br]{%s}}\n",
             x1,y0 + i*vspc + spot_height*name_up_factor,
             names[p.winner(p.rounds()-r-1,(1 << r+1) - i - 1)].c_str());
    }

    // recurse
    bracket(p,r-1,x0+spot_width,x1-spot_width,y0+vspc/2,2*vspc);
  }
}
           
//
// one set of picks
//
void do_one_set(char *line) {
  Picks X(rounds_from_teams(names.size()));

  istringstream s(line);
  try {
    read_depth(X,s);
  } catch (TP_read_error& err) {
    err.display();
    return;
  }

  spot_height = 2 * pic_height / X.teams();
  spot_width  = pic_width / (2*X.rounds() - 1 + lr_gap_factor);

  printf("\\begin{picture}(%.2f,%.2f)\n",pic_width,pic_height);
  printf("    \\tiny \\sffamily \\thinlines\n");

  bracket(X,X.rounds()-1,0,pic_width,0,spot_height);

  printf("\\end{picture}\n");
}

//
// main
//
main(int argc, char *argv[]) {
  // Parse arguments

  char opt;
  while ((opt = getopt(argc,argv,"w:h:")) != -1)
    switch (opt) {
    case 'h':
      pic_height = atof(optarg);
      break;
    case 'w':
      pic_width = atof(optarg);
      break;
    default:
      usage();
    }
  if (optind != argc-1) usage();

  ifstream nf(argv[optind]);
  if (!nf) {
    perror(argv[optind]);
    exit(0);
  }

  try {
    title = read_namesfile(nf,names);
  } catch (TP_read_error& err) {
    err.display();
    exit(1);
  }

  char line[256];
  while(cin.getline(line,256)) {
    do_one_set(line);
  }
}

