/*
 * Copyright (C) 2005 Bryan Clair
 *
 * This file is part of CLOP.
 *
 * CLOP is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * CLOP is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with CLOP; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *
 *
 * StatTest
 *
 *   Classes for running quick and dirty checks of
 *   calculated statistics.
 */

#include <iostream>
#include <iomanip>
#include <unistd.h>

#include "StatTest.h"
using namespace std;

// field width & precision for output
#define FW 8
#define PREC 5

/**************************************************/
/* Trials Class                                   */
/**************************************************/

void t_header() {
  cout << setprecision(PREC);
  cout << setw(FW) << "TRIALS" << " ";
  cout << setw(FW) << "MEAN" << " ";
  cout << setw(FW) << "VAR" << " ";
  cout << setw(FW) << "SD";
  cout << endl;
}

void t_stats(double mean, double var) {
  cout << setw(FW) << mean << " ";
  cout << setw(FW) << var << " ";
  cout << setw(FW) << sqrt(var);
  cout << endl;
}

Trials::Trials() {
  t_header();

  num = 0;
  mean = 0;
  sumsq = 0;
  last = time(NULL);
}

Trials::Trials(double cmean, double cvar) {
  t_header();
  cout << setw(FW) << "(calc)" << " ";
  t_stats(cmean,cvar);

  num = 0;
  mean = 0;
  sumsq = 0;
  last = time(NULL);
}

void Trials::display() const {
  cout << setw(FW) << num << " ";
  t_stats(mean,sumsq/(num-1));
}

void Trials::add_point(double x) {
  // This on-the-fly algorithm is reportedly due to R. Jennrich
  double dev;

  num++;
  dev = x-mean;
  mean += dev/(double)num;
  sumsq += dev*(x-mean);
  
  // Report results about every second
  if ((num % 100 == 0) && (time(NULL)!= last)) {
    display();
    last = time(NULL);
  }
}

/**************************************************/
/* CoTrials Class                                 */
/**************************************************/

void ct_header() {
  cout << setprecision(PREC);
  cout << setw(FW) << "TRIALS" << " ";
  cout << setw(FW) << "MEAN1" << " ";
  cout << setw(FW) << "VAR1" << " ";
  cout << setw(FW) << "SD1" << " ";
  cout << setw(FW) << "MEAN2" << " ";
  cout << setw(FW) << "VAR2" << " ";
  cout << setw(FW) << "SD2" << " ";
  cout << setw(FW) << "COVAR" << " ";
  cout << setw(FW) << "CORR";
  cout << endl;
}

void ct_stats(double mean1, double var1,
	      double mean2, double var2,
	      double covar) {
  cout << setw(FW) << mean1 << " ";
  cout << setw(FW) << var1 << " ";
  cout << setw(FW) << sqrt(var1) << " ";
  cout << setw(FW) << mean2 << " ";
  cout << setw(FW) << var2 << " ";
  cout << setw(FW) << sqrt(var2) << " ";
  cout << setw(FW) << covar << " ";
  cout << setw(FW) << covar/(sqrt(var1)*sqrt(var2));
  cout << endl;
}
  
CoTrials::CoTrials() {
  num = 0;
  meanx = meany = 0;
  sumxsq = sumysq = 0;
  sumxy = 0;
  last = time(NULL);

  ct_header();
}

CoTrials::CoTrials(double cmean1, double cvar1,
		   double cmean2, double cvar2,
		   double ccovar) {

  ct_header();
  cout << setw(FW) << "(calc)" << " ";
  ct_stats(cmean1,cvar1,cmean2,cvar2,ccovar);

  num = 0;
  meanx = meany = 0;
  sumxsq = sumysq = 0;
  sumxy = 0;
  last = time(NULL);
}

void CoTrials::display() const {
  cout << setw(FW) << num << " ";
  ct_stats(meanx,sumxsq/(num-1),meany,sumysq/(num-1),sumxy/(num-1));
}

void CoTrials::add_point(double x, double y) {
  double devx, devy;

  num++;
  devx = x-meanx;
  meanx += devx/(double)num;
  sumxsq += devx*(x-meanx);
  devy = y-meany;
  meany += devy/(double)num;
  sumysq += devy*(y-meany);
  
  sumxy += devx * (y - meany);   // = (x - meanx) * devy
  
  // Report results about every second
  if ((num % 100 == 0) && (time(NULL)!= last)) {
    display();
    last = time(NULL);
  }
}
