/*
 * Copyright (C) 2005 Bryan Clair
 *
 * This file is part of CLOP.
 *
 * CLOP is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * CLOP is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with CLOP; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *
 */	#include <math.h>
#include <limits>
#include <iostream>
#include <iomanip>
#include <cassert>
#include <gsl/gsl_math.h>
#include <gsl/gsl_sf.h>
#include <gsl/gsl_integration.h>

using namespace std;

  // gsl_sf_erf_Z(x) computes the Gaussian probability density function
  // Z(x) = (1/\sqrt{2\pi}) \exp(-x^2/2).   


//
// double log_CDF(double x)
//   Computes log(CDF(x)) with accuracy for all values of x
//   CDF is the Cumulative Distribution Function for a normalized random variable
//   CDF(x) = \Phi(x) = \int_{-infty}^{x} Z(x) = .5*(1 + erf(x/sqrt(2))
//          = 1 - Q(x)
//   0 < CDF(x) < 1, so -infty < log(CDF(x)) < 0
//
double log_CDF(double x) {
  if (x > 0)
    return gsl_sf_log_1plusx(-erfc(x/M_SQRT2)/2);
  else // (x <= 0)
    return gsl_sf_log_erfc(-x/M_SQRT2) - M_LN2;
}

//
// double log_exp_x_m1(double x)
//   Computes log(exp(x)-1) for x > 0 in a way that is stable for all x.
//
double log_exp_x_m1(double x) {
  static const double big = -log(numeric_limits<double>::epsilon());
  if (x > big)
    return x;
  else
    return log(gsl_sf_expm1(x));
  // which is log(x) for x < epsilon.
}

//
// double win_distribution(double N, double y, double x)
//    For y < x, computes 
//                      N+1          N+1
//                CDF(x)    -  CDF(y) 
//                ----------------------
//                   CDF(x) - CDF(y)
//
double win_distribution(double N, double y, double x) {
  assert(x > y);

  double lCDFx = log_CDF(x), lCDFy = log_CDF(y);

  if (lCDFx - lCDFy < numeric_limits<double>::epsilon()/N) {
    cout << "small!";
    return (N+1)*exp(N*lCDFy);
  }
  else
    return exp(N*lCDFy
	       + log_exp_x_m1((N+1)*(lCDFx - lCDFy))
	       - log_exp_x_m1(lCDFx - lCDFy) );
}

// pooltie
//   Computes    N+1   N+1
//              x  -  y        n   k    N-k
//    f(x,y) =  -------  = sum    x  * y
//               x - y       k=0
//
double pooltie(int N, double x, double y)
{
  //  cout << "pooltie(" << N << "," << x << "," << y << ")\n";

  //
  // If x and y are very close together, we replace f with its
  // derivative
  //
  if (gsl_fcmp(x,y,numeric_limits<double>::epsilon()) == 0)
    return (N+1)*pow(x,N);
  else
    return (pow(x,N+1) - pow(y,N+1))/(x - y);
}
double win_distribution_bad(int N, double y, double x) {
  assert(x > y);
  return pooltie(N,1-gsl_sf_erf_Q(x),1-gsl_sf_erf_Q(y));
}

main() {
  double a = 5, b = 6;
  double x;
  int i, N = 30;
  cout << setprecision(40);

  cout << "radix: " << numeric_limits<double>::radix << endl;
  cout << "exponent range: ";
  cout << numeric_limits<double>::min_exponent << "...";
  cout << numeric_limits<double>::max_exponent << endl;
  cout << "epsilon: " << numeric_limits<double>::epsilon() << endl;
  cout << "mantissa digits (base 2): " << numeric_limits<double>::digits << endl;


  double delta = .000001;
  for (i = 1, x = -5; i < 100; i++, x = x+1/50.) 
    cout << x << ": " << log_CDF(x) << "   " << win_distribution(1000000,-x-delta,-x) << "    bad: " <<
      win_distribution_bad(1000000,-x-delta,-x) << endl;
}
  
