/*
 * Copyright (C) 2005 Bryan Clair
 *
 * This file is part of CLOP.
 *
 * CLOP is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * CLOP is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with CLOP; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *
 *
 * smoothtest.C
 *
 *  Find expected return for some canonical bets in an NCAA randomized
 *  pool against varying numbers of opponents.
 */

#include <iostream>
#include <iomanip>
#include "TourneyPack.h"
#include "TourneyApp.h"
#include "StatTest.h"
#include "Normal.h"

using namespace std;

main(int argc, char *argv[])
{
  //
  // Parse arguments
  //
  round_t R = TourneyApp_initialize(argc,argv);

  //
  // A set of actual probabilites
  //
  //  ProbSuite a0(R,PS_NCAA_seed_history);
  //  ProbSuite a = a0.random_variation();
  ProbSuite a = PS_random(R);

  cout << "ACTUAL probabilities: random" << endl;

  //
  // Some perceptions
  //
  ProbSuite p = PS_random(R);
  cout << "PERCEIVED VALUES: random" << endl;

  //
  // Some interesting picks
  //
  Picks favs = a.favorites();
  cout << "\nFAVORITES:" << endl;
  favs.display();

  double ret;
  Picks likely = a.most_likely(&ret);
  cout << "\nLIKELY:" << endl;
  cout << "(with probability " << ret << ")" << endl;
  likely.display();
  
  Picks maxex = a.max_expected_score(&ret);
  cout << "\nMAX EXPECTED SCORE:" << endl;
  cout << "(with expected score " << ret << ")\n";
  maxex.display();

  Picks limit = best_for_large_pool(a,p,&ret);
  cout << "\nBEST PICKS IN A VERY LARGE POOL:" << endl;
  cout << "(with an expected return of " << ret << " on a bet of 1)\n";
  limit.display();
  
  Picks guess = p.random_picks();
  cout << "\nA RANDOM GUESS USING THESE PERCEIVED VALUES:" << endl;
  guess.display();

  //
  // General info about rounds & worth
  //
  cout << "\nROUNDS ARE WORTH: " << worth(1);
  for (round_t k = 2; k <= R; k++) cout << ", " << worth(k);
  cout << endl;
  cout << "MAX SCORE POSSIBLE: " << worth_max(R) << endl;
  cout << "MEAN FOR RANDOM PICKS: " << mean_for_random_picks(R) << endl;

  cout << "\nPROBABILITIES OF HAPPENING 100%:" << endl;
  cout << "maxexp:" << probability(a,maxex) << endl;
  cout << "favs  :" << probability(a,favs) << endl;
  cout << "likely:" << probability(a,likely) << endl;
  cout << "guess :" << probability(a,guess) << endl;
  cout << "limit :" << probability(a,limit) << endl;

  cout << "\nEXPECTED SCORE:" << endl;
  cout << "maxex :" << mean(a,maxex) << endl;
  cout << "favs  :" << mean(a,favs) << endl;
  cout << "likely:" << mean(a,likely) << endl;
  cout << "guess :" << mean(a,guess) << endl;
  cout << "limit :" << mean(a,limit) << endl;
  cout << "oppts :" << mean(a,p) << endl;

  cout << "\nOPPONENTS:" << endl;
  cout << "mean " << mean(a,p) << endl;
  cout << "variance " << variance(a,p) << endl;
  cout << "covariance " << covariance(a,p,p) << endl;

  cout << "\nAPPROXIMATE RETURNS ON BETS:" << endl;
  cout << setw(12) << "#oppts";
  cout << setw(15) << "copycat";
  cout << setw(15) << "maxexp";
  cout << setw(15) << "favs";
  cout << setw(15) << "guess";
  cout << setw(15) << "limit";
  cout << endl;

  Normal_integrator nit;

  int N;
  for (N=1; N <= 1000000000; N = N*10) {
    cout << setw(12) << N << setw(15);

    cout << nit.expect_integrate(N,mean(a,p),variance(a,p),
				 mean(a,p),variance(a,p),
				 covariance(a,p,p), covariance(a,p,p));
    cout << setw(15);
    cout << nit.expect_integrate(N,mean(a,p),variance(a,p),
				 mean(a,maxex),variance(a,maxex),
				 covariance(a,p,p), covariance(a,p,maxex));
    cout << setw(15);
    cout << nit.expect_integrate(N,mean(a,p),variance(a,p),
				 mean(a,favs),variance(a,favs),
				 covariance(a,p,p), covariance(a,p,favs));
    cout << setw(15);
    cout << nit.expect_integrate(N,mean(a,p),variance(a,p),
				 mean(a,guess),variance(a,guess),
				 covariance(a,p,p), covariance(a,p,guess));
    cout << setw(15);
    cout << nit.expect_integrate(N,mean(a,p),variance(a,p),
				 mean(a,limit),variance(a,limit),
				 covariance(a,p,p), covariance(a,p,limit));
    cout << endl;
  }

}
