/*
 * topponents.C
 *
 *   Given a file of opponent picks, calculate statistics about opponents
 *   for each outcome on stdin.
 *
 * usage: topponents [-qE] namesfile actualfile perceivedfile opponentfile
 *   -q  quiet
 *   -s  show stats for each outcome
 *   -t  show teams for each outcome
 *   -E  use ESPN worth
 *
 * Copyright (C) 2005 Bryan Clair
 *
 * This file is part of CLOP.
 *
 * CLOP is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * CLOP is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with CLOP; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *
 * 9/29/16 Fixing obsolete code so it will compile.
 */

#include <iostream>
#include <fstream>
#include <sstream>
#include <vector>
#include <stdlib.h>
#include <unistd.h>

#include "TourneyPack.h"
#include "TourneyPool.h"
#include "TourneyFile.h"

using namespace std;

//
// usage
//
void usage(char *msg=NULL) {
  cerr << "usage: topponents [-qstE] namesfile actualfile perceivedfile "
       << endl
       << "                opponentfile"
       << endl;
  if (msg) cerr << "       " << msg << endl;
  exit(1);
}

//
// global pool, opponent pick set, count of picks
//
TourneyPool *pool;
vector<Picks> pickset;
int *counts;
int maxs, step;  // score max and step

//
// default arguments
//
bool quiet = false;
bool showpix = false;
bool showstats = false;

//
// one set of winner picks
//
void read_one_set(char *line) {
  Picks winners(pool->rounds());
  istringstream s(line);
  try {
    read_depth(winners,s);
  } catch (TP_read_error& err) {
    err.display();
    return;
  }

  if (!quiet) {
    if (showpix) pool->display_picks(winners);
    if (showstats) pool->display_stats(winners);
  }

  for (int i=0; i<pickset.size(); i++) {
    counts[score(winners,pickset[i])/step]++;
  }

}

//
// main
//
main(int argc, char *argv[]) {
  // Parse arguments

  char opt;
  while ((opt = getopt(argc,argv,"Eqts")) != -1)
    switch (opt) {
    case 'q':
      quiet = true;
      break;
    case 't':
      showpix = true;
      break;
    case 's':
      showstats = true;
      break;
    case 'E':
      worth_function_set(worth_ESPN);
      break;
    default:
      usage();
    }
  if (optind != argc-4) usage();

  // Read pool data
  try {
    pool = read_pool(argv+optind);
  } catch (TP_read_error& err) {
    err.display();
    exit(1);
  }
  // Print header
  if (!quiet) {
    pool->display_summary();
    pool->display_opponent_stats();
  }


  // Read opponent picks
  fstream pfile(argv[optind+3]);
  Picks X(pool->rounds());
  while (pfile) {
    try {
      read_depth(X,pfile);
      pickset.push_back(X);
    } catch (TP_read_error& err) {
      if (!pfile.eof()) {
	err.display();
	exit(1);
      }
    }
  }
  if (!quiet) {
    cout << "Read " << pickset.size() << " picks from file " 
	 << argv[optind+3] << endl;
  }

  // Initialize score histogram bins
  step = worth_gcd(pool->rounds());
  maxs = worth_max(pool->rounds())/step;
  counts = new int[maxs + 1];
  for (int i=0; i<=maxs; i++) counts[i]=0;

  // Loop through outcome picks read from stdin
  char line[256];
  while(cin.getline(line,256)) {
    read_one_set(line);
  }

  // Display results
  for (int i=0; i<=maxs; i++)
    cout << i*step << "\t" << counts[i] << endl;
}

