/*
 * tseek.C
 *
 *    Hill-climbing seek for good bets in a tournament.
 *
 * usage: tseek [-qvE] [-t<trials>] [-n<competitors>] [-s<seed>] namesfile..
 *             actualfile perceivedfile
 *   -q : quiet.. only display final best picks (so better use -t also)
 *   -v : debug.. print intermediate results
 *   -E : use ESPN worth
 *
 * Copyright (C) 2005 Bryan Clair
 *
 * This file is part of CLOP.
 *
 * CLOP is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * CLOP is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with CLOP; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *
 * 9/29/16 Fixing obsolete code so it will compile.
 */

#include <iostream>
#include <fstream>
#include <time.h>
#include <unistd.h>

#include "TourneyPack.h"
#include "TourneyPool.h"
#include "TourneyFile.h"
using namespace std;

//
// global pool
//
TourneyPool *pool;

//
// default arguments
//
long competitors = 1000;
bool quiet = false;
bool verbose = false;
int trials = -1;

//
// usage
//
void usage(const char *msg=NULL) {
  cerr << "usage: tseek [-qvE] [-t<trials>] [-n<competitors>] [-s<seed>] namesfile.."
       << endl
       << "                actualfile perceivedfile"
       << endl;
  if (msg) cerr << "       " << msg << endl;
  exit(1);
}

//
// greedy_seek
//    Beginning with start Picks, move to best neighbor until
//    a local max is reached.  Returns the local max Picks
//    and stores the local max value in expected_return
//
Picks greedy_seek(const Picks &start, double *expected_return,
		  Normal_integrator& nit)
{
  // cur_picks is the best picks found so far,
  // cur_best is the best expected return
  //
  Picks cur_picks = start;

  double cur_val = pool->smooth_expected(competitors, cur_picks, &nit);

  if (verbose) {
    cout << "working on ";
    write_depth(cur_picks);
    cout << " (" << cur_val << ")" << endl;
  }

  double nbr_val, best_nbr_val = 0;
  Picks nbr_picks = start;
  int i, best_nbr = 0, nbrs = start.num_neighbors();

  while (true) {
    // Find best neighbor
    for (i = 0; i < nbrs; i++) {
      nbr_picks = cur_picks.neighbor(i);
      nbr_val = pool->smooth_expected(competitors, nbr_picks, &nit);

      if (nbr_val > best_nbr_val) {
	best_nbr_val = nbr_val;
	best_nbr = i;
      }
    }

    // Move to best neighbor or bail out
    if (best_nbr_val > cur_val) {
      cur_picks = cur_picks.neighbor(best_nbr);
      cur_val = best_nbr_val;
      if (verbose) {
	cout << "  moved to ";
	write_depth(cur_picks);
	cout << " (" << cur_val << ")" << endl;
      }
    } else break;
  }

  *expected_return = cur_val;
  return cur_picks;
}

//
// greedy_randomized_search
//    Apply greedy_seek to trials random picks.
//    Return the best Picks found, and optionally store the best
//    value found in expected_return.
//
Picks greedy_randomized_search(double *expected_return)
{
  Normal_integrator nit;
  double best_val = -1, found_val;
  Picks best_picks(pool->rounds());
  Picks found(pool->rounds());

  time_t start = time(NULL);
  time_t lastreport = start;
  int curtrial = 0;

  while (curtrial != trials) {
    found = greedy_seek(random_picks(pool->rounds()), &found_val, nit);
    if (found_val > best_val) {
      best_val = found_val;
      best_picks = found;
      // display new best picks
      cout << found_val << '\t';
      write_depth(found);
      cout << endl;
    }

    curtrial++;

    if (time(NULL) - lastreport >= 60) {
      if (!quiet) {
	cout << curtrial << " trials, "
	     << ((time(NULL)-start)/60)
	     << " minutes elapsed." << endl;
      }
      lastreport = time(NULL);
    }
  }

  *expected_return = best_val;
  return best_picks;
}

//
// main
//
main(int argc, char *argv[])
{
  long seed = time(NULL);

  //
  // Parse arguments
  //
  char opt;
  while ((opt = getopt(argc,argv,"Eqvs:t:n:")) != -1)
    switch (opt) {
    case 'n':
      competitors = atol(optarg);
      break;
    case 't':
      trials = atoi(optarg);
      break;
    case 's':
      seed = atol(optarg);
      break;
    case 'q':
      quiet = true;
      break;
    case 'v':
      verbose = true;
      break;
    case 'E':
      worth_function_set(worth_ESPN);
      break;
    default:
      usage();
    }
  if (competitors <= 0) usage("# of competitors must be > 0");
  if (optind != argc-3) usage();

  srandom(seed);

  try {
    pool = read_pool(argv+optind);
  } catch (TP_read_error& err) {
    err.display();
    exit(1);
  }

  // Print header
  if (!quiet) {
    pool->display_summary();
    cout << competitors << " competitors." << endl;
  }

  double expected_return;
  Picks best = greedy_randomized_search(&expected_return);

  if (!quiet) {
    cout << "best found:" << endl;
  }
  cout << expected_return << '\t';
  write_depth(best);
  cout << endl;
}
