/*
 * tsim.C
 *
 * Simulate tournaments.
 *
 * Performs <trials> simulations for each set of picks Y read on standard in.
 *
 * Each trial chooses n competitor picks, either randomly using perceived
 * probablities or by selecting from the opponentpicks file if provided.
 * Each trial chooses winners using actual probabilities and calculates
 * the score and winnings for picks Y.
 * After all trials are finished, the summary results for picks Y
 * are displayed.
 *
 * usage: tsim [-qntEs] namesfile actualfile perceivedfile [opponentfile]
 *   -q  quiet
 *   -n  number of opponents per trial
 *   -t  number of trials
 *   -E  use ESPN worth
 *   -s  seed for random number generator (default is current time)
 *
 * Copyright (C) 2005 Bryan Clair
 *
 * This file is part of CLOP.
 *
 * CLOP is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * CLOP is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with CLOP; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *
 * Revisions:
 * 6/27/06: Added the ability to simulate using opponent picks from a file
 * 9/29/16 Fixing obsolete code so it will compile.
 */

#include <iostream>
#include <sstream>
#include <fstream>
#include <stdlib.h>
#include <time.h>
#include <vector>
#include <unistd.h>

#include "TourneyPack.h"
#include "TourneyPool.h"
#include "TourneyFile.h"

using namespace std;

//
// usage
//
void usage(const char *msg=NULL) {
  cerr << "usage: tsim [-qntEs] namesfile actualfile perceivedfile [opponentfile]"
       << endl;
  cerr << "   -q  quiet" << endl;
  cerr << "   -n  number of opponents per trial" << endl;
  cerr << "   -t  number of trials" << endl;
  cerr << "   -E  use ESPN worth" << endl;
  cerr << "   -s  seed for random number generator (default is current time)" << endl;

  if (msg) cerr << "       " << msg << endl;
  exit(1);
}

//
// global pool, opponent pick data
//
TourneyPool *pool;
vector<Picks> pickset;
int total_score;

//
// default arguments
//
long competitors = 100;
long trials = 10000;
bool quiet = false;
bool simulated_opponents = true;

//
// simulate_once
//
double simulate_once(Picks Y) {
  Picks winners = pool->actual().random_picks();

  int myscore = score(winners,Y);
  total_score += myscore;

  int tied = 1;
  int cscore;
  for (int i=0; i < competitors; i++) {
    if (simulated_opponents) {
      cscore = score(winners,pool->perceived().random_picks());
    } else {
      cscore = score(winners,pickset[random() % pickset.size()]);
    }

    if (cscore > myscore) {
      return 0;
    }
    if (cscore == myscore) tied++;
  }

  return competitors/((double) tied);
}

//
// one set of picks
//
void do_one_set(char *line) {
  Picks Y(pool->rounds());
  istringstream s(line);
  try {
    read_depth(Y,s);
  } catch (TP_read_error& err) {
    err.display();
    return;
  }

  total_score = 0;

  double total=0, val;
  int wins=0;

  for (int t=0; t < trials; t++) {
    val = simulate_once(Y);
    if (val > 0) wins++;
    total += val;
  }
  
  cout << mean(pool->actual(),Y) << "\t";
  cout << pool->smooth_expected(competitors,Y) << "\t";
  cout << total_score/((double)trials) << "\t";
  cout << total/trials << "\t";
  cout << wins << endl;
}

//
// main
//
main(int argc, char *argv[]) {
  long seed = time(NULL);

  // Parse arguments

  char opt;
  while ((opt = getopt(argc,argv,"Eqs:t:n:")) != -1)
    switch (opt) {
    case 'q':
      quiet = true;
      break;
    case 'n':
      competitors = atol(optarg);
      break;
    case 's':
      seed = atol(optarg);
      break;
    case 't':
      trials = atol(optarg);
      break;
    case 'E':
      worth_function_set(worth_ESPN);
      break;
    default:
      usage();
    }
  if (competitors <= 0) usage("# of competitors must be > 0");
  switch (argc - optind) {
  case 4:
    simulated_opponents = false;
    break;
  case 3:
    simulated_opponents = true;
    break;
  default:
    usage();
  }

  srandom(seed);

  try {
    pool = read_pool(argv+optind);
  } catch (TP_read_error& err) {
    err.display();
    exit(1);
  }

  // Pool read correctly - brag about it
  if (!quiet) {
    pool->display_summary();
    cout << endl;
  }

  if (simulated_opponents) {
    if (!quiet) cout << "Using simulated opponents." << endl << endl;
  } else {
    // Read opponent picks
    fstream pfile(argv[optind+3]);
    Picks X(pool->rounds());
    while (pfile) {
      try {
	read_depth(X,pfile);
	pickset.push_back(X);
      } catch (TP_read_error& err) {
	if (!pfile.eof()) {
	err.display();
	exit(1);
	}
      }
    }
    if (!quiet) {
      cout << "Read " << pickset.size() << " picks from file " 
	   << argv[optind+3] << endl << endl;
    }
  }

  // Print header
  if (!quiet) {
    cout << competitors << " competitors." << endl;
    cout << trials << " trials." << endl;
    cout << endl;
    cout << "EXPECTED\tSIMULATED" << endl;
    cout << "SCORE\tRETURN\tSCORE\tRETURN\tWINS" << endl;
  }

  char line[256];
  while(cin.getline(line,256)) {
    do_one_set(line);
  }
}

