/*
 * tstats.C
 *
 *    Calculate stats for picks read from stdin.
 *
 * usage: tstats [-eqstEP] [-n<competitors>] namesfile actualfile perceivedfile
 *   -q  quiet
 *   -s  don't show stats
 *   -e  don't show expected
 *   -t  don't show teams
 *   -P  show perceived probability only
 *   -E  use ESPN worth
 *
 * Copyright (C) 2005 Bryan Clair
 *
 * This file is part of CLOP.
 *
 * CLOP is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * CLOP is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with CLOP; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *
 * 9/29/16 Fixing obsolete code so it will compile.
 */

#include <iostream>
#include <sstream>
#include <stdlib.h>
#include <unistd.h>

#include "TourneyPack.h"
#include "TourneyPool.h"
#include "TourneyFile.h"

using namespace std;

//
// usage
//
void usage(const char *msg=NULL) {
  cerr << "usage: tstats [-eqstEP] [-n<competitors>] namesfile actualfile perceivedfile"
       << endl;
  cerr << "   -q  quiet\n"
       << "   -s  don't show stats\n"
       << "   -e  don't show expected\n"
       << "   -t  don't show teams\n"
       << "   -P  show perceived probability only\n"
       << "   -E  use ESPN worth\n";
  if (msg) cerr << "       " << msg << endl;
  exit(1);
}

//
// global pool
//
TourneyPool *pool;

//
// default arguments
//
long competitors = 1000;
bool showpp = false;
bool showexp = true;
bool showpix = true;
bool showstats = true;
bool quiet = false;

//
// one set of picks
//
void do_one_set(char *line) {
  Picks X(pool->rounds());
  istringstream s(line);
  try {
    read_depth(X,s);
  } catch (TP_read_error& err) {
    err.display();
    return;
  }


  if (showpix) pool->display_picks(X);

  if (showexp) {
    cout << "expected return: " << pool->smooth_expected(competitors,X) << endl;
  }
  if (showstats) pool->display_stats(X);

  if (showpp)
    cout << probability(pool->perceived(),X) << endl;

}

//
// main
//
main(int argc, char *argv[]) {
  // Parse arguments

  char opt;
  while ((opt = getopt(argc,argv,"EqsetPn:")) != -1)
    switch (opt) {
    case 'q':
      quiet = true;
      break;
    case 's':
      showstats = false;
      break;
    case 'e':
      showexp = false;
      break;
    case 't':
      showpix = false;
      break;
    case 'n':
      competitors = atol(optarg);
      break;
    case 'P':
      showstats = false;
      showexp = false;
      showpix = false;
      showpp = true;
      break;
    case 'E':
      worth_function_set(worth_ESPN);
      break;
    default:
      usage();
    }
  if (competitors <= 0) usage("# of competitors must be > 0");
  if (optind != argc-3) usage();

  try {
    pool = read_pool(argv+optind);
  } catch (TP_read_error& err) {
    err.display();
    exit(1);
  }

  // Print header
  // Print header
  if (!quiet) {
    pool->display_summary();
    cout << competitors << " competitors." << endl;
    pool->display_opponent_stats();
  }

  char line[256];
  while(cin.getline(line,256)) {
    do_one_set(line);
  }
}

